// $(document).ready(() => {
//   if (
//     typeof window.prestashop !== 'undefined' &&
//     window.prestashop.component &&
//     window.prestashop.component.GridExtensions
//   ) {
//     // Create our own Grid instance for the order grid
//     // Use the correct path to Grid constructor
//     const orderGrid = new window.prestashop.component.Grid('order');
//
//     // Add our extension
//     if (window.prestashop.component.GridExtensions.AjaxBulkActionExtension) {
//       orderGrid.addExtension(new window.prestashop.component.GridExtensions.AjaxBulkActionExtension());
//       console.log('AjaxBulkActionExtension added to Orders grid');
//     } else {
//       console.error('AjaxBulkActionExtension not found');
//     }
//   }
// });

/**
 * Custom Ajax Bulk Action Extension that supports direct URLs
 */
/**
 * Custom Ajax Bulk Action implementation for Orders Grid
 */
$(document).ready(function() {
  // Attach to the document and filter for clicks on the order grid's bulk ajax buttons
  $(document).on('click', '.js-dupa', function(event) {
    event.preventDefault();
    event.stopPropagation();

    const $ajaxButton = $(event.currentTarget);
    const $checkboxes = $('.js-bulk-action-checkbox:checked');
    const selectedIds = $checkboxes.get().map((checkbox) => checkbox.value);

    if (selectedIds.length === 0) {
      return;
    }

    const confirmBulkAction = $ajaxButton.data('confirmBulkAction') !== false;
    const bulkAction = $ajaxButton.data('bulkAction') || 'bulk-action';
    const progressionTitle = $ajaxButton.data('progressTitle') || 'Processing items';
    const confirmTitle = $ajaxButton.data('confirmTitle') || 'Apply modifications';
    const progressionMessage = $ajaxButton.data('progressMessage') || 'Processing %done% / %total% items...';

    if (confirmBulkAction) {
      if (confirm(progressionTitle.replace('%total%', selectedIds.length))) {
        processAjaxBulkAction($ajaxButton, selectedIds);
      }
    } else {
      processAjaxBulkAction($ajaxButton, selectedIds);
    }
  });

  /**
   * Create a simple progress modal since PrestaShop's component is not available
   */
  function createSimpleProgressModal(title, initialMessage) {
    // Create modal elements
    const $modalOverlay = $('<div class="modal-backdrop show"></div>');
    const $modalContainer = $(`
      <div class="modal fade show d-block" tabindex="-1" role="dialog">
        <div class="modal-dialog" role="document">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">${title}</h5>
            </div>
            <div class="modal-body">
              <p class="progress-message">${initialMessage}</p>
              <div class="progress">
                <div class="progress-bar" role="progressbar" style="width: 0%;" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100">0%</div>
              </div>
              <div class="mt-3 error-messages"></div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-secondary abort-button">Abort</button>
              <button type="button" class="btn btn-primary close-button" style="display: none;">Close</button>
            </div>
          </div>
        </div>
      </div>
    `);

    // Append to body
    $('body').append($modalOverlay).append($modalContainer);

    // Modal API
    return {
      updateProgress: function(done, total) {
        const percent = Math.round((done / total) * 100);
        $modalContainer.find('.progress-bar').css('width', percent + '%').attr('aria-valuenow', percent).text(percent + '%');
        $modalContainer.find('.progress-message').text(initialMessage.replace('%done%', done).replace('%total%', total));
      },
      addError: function(errorMessage) {
        $modalContainer.find('.error-messages').append(`<div class="alert alert-danger">${errorMessage}</div>`);
      },
      completeProgress: function() {
        $modalContainer.find('.abort-button').hide();
        $modalContainer.find('.close-button').show();
      },
      onAbort: function(callback) {
        $modalContainer.find('.abort-button').on('click', function() {
          callback();
          $(this).prop('disabled', true).text('Aborting...');
        });
      },
      onClose: function(callback) {
        $modalContainer.find('.close-button').on('click', function() {
          $modalOverlay.remove();
          $modalContainer.remove();
          callback();
        });
      },
      close: function() {
        $modalOverlay.remove();
        $modalContainer.remove();
      }
    };
  }

  /**
   * Process the AJAX bulk action with progress tracking
   */
  function processAjaxBulkAction($ajaxButton, selectedIds) {
    const bulkChunkSize = parseInt($ajaxButton.data('requestBulkChunkSize') || 10);
    const reloadAfterBulk = $ajaxButton.data('reloadAfterBulk') !== false;
    const progressionTitle = $ajaxButton.data('progressTitle') || 'Processing items';
    const progressionMessage = $ajaxButton.data('progressMessage') || 'Processing %done% / %total% items...';

    // Create simple progress modal
    const modal = createSimpleProgressModal(
      progressionTitle.replace('%total%', selectedIds.length),
      progressionMessage.replace('%done%', '0').replace('%total%', selectedIds.length)
    );

    // Setup abort controller
    const abortController = new AbortController();
    modal.onAbort(() => {
      abortController.abort();
    });

    modal.onClose(() => {
      if (reloadAfterBulk) {
        window.location.reload();
      }
    });

    // Chunk the IDs
    const chunkedIds = [];
    for (let i = 0; i < selectedIds.length; i += bulkChunkSize) {
      chunkedIds.push(selectedIds.slice(i, i + bulkChunkSize));
    }

    let doneCount = 0;
    let errors = [];

    // Process each chunk sequentially
    processChunks(chunkedIds, 0);

    function processChunks(chunks, index) {
      if (index >= chunks.length) {
        // All chunks processed
        modal.completeProgress();
        return;
      }

      const chunk = chunks[index];
      callAjaxAction($ajaxButton, chunk, abortController.signal)
        .then(response => {
          if (abortController.signal.aborted) {
            return;
          }

          doneCount += chunk.length;
          modal.updateProgress(doneCount, selectedIds.length);

          return response.json().catch(() => null);
        })
        .then(data => {
          if (data && (data.errors || data.error || data.message)) {
            if (Array.isArray(data.errors)) {
              data.errors.forEach(error => {
                errors.push(error);
                modal.addError(error);
              });
            } else {
              const error = data.errors || data.error || data.message || 'Unknown error';
              errors.push(error);
              modal.addError(error);
            }
          }

          // Process next chunk
          processChunks(chunks, index + 1);
        })
        .catch(error => {
          if (!abortController.signal.aborted) {
            const errorMessage = error.message || 'Request failed';
            errors.push(errorMessage);
            modal.addError(errorMessage);
          }

          // Continue with next chunk despite error
          processChunks(chunks, index + 1);
        });
    }
  }

  /**
   * Call the AJAX action with a direct URL
   */
  function callAjaxAction($ajaxButton, chunkIds, abortSignal) {
    const requestParamName = $ajaxButton.data('requestParamName') || 'bulk_ids';
    const routeParams = $ajaxButton.data('routeParams') || {};
    const routeMethod = $ajaxButton.data('routeMethod') || 'POST';

    // Get the direct URL from data-ajax-url
    let url = $ajaxButton.data('ajax-url');
    if (!url) {
      console.error('No ajax-url provided in data attributes');
      return Promise.reject(new Error('No URL provided'));
    }

    // Create FormData
    const formData = new FormData();
    chunkIds.forEach((chunkId, index) => {
      formData.append(`${requestParamName}[${index}]`, chunkId);
    });

    // Handle route parameters
    const urlObj = new URL(url, window.location.origin);
    Object.entries(routeParams).forEach(([key, value]) => {
      urlObj.searchParams.append(key, value);
    });

    // Determine HTTP method
    let requestMethod;
    switch (routeMethod.toUpperCase()) {
      case 'PATCH':
      case 'DELETE':
        requestMethod = 'POST';
        break;
      default:
        requestMethod = routeMethod;
    }

    // Make the fetch request
    return fetch(urlObj.toString(), {
      method: requestMethod,
      body: formData,
      headers: {
        '_method': routeMethod
      },
      signal: abortSignal
    });
  }
});
