/**
 * Form handler for Orlen Paczka module
 */
import { showErrorMessage, hideErrorMessage, hideAllErrors, createGeneralErrorDiv } from '../utils/form-utils.js';
import { sendFormData, fetchAvailableDates, fetchAvailableHours } from '../services/api-service.js';

/**
 * Reset form to initial state
 * @param {HTMLFormElement} form - The form to reset
 */
export const resetFormValues = (form) => {
  form.reset();
  hideAllErrors(form);
  
  // Remove error alerts
  const errorAlerts = form.querySelectorAll('.alert.alert-danger');
  errorAlerts.forEach(alert => alert.remove());

  // Reset select elements
  const dateSelect = form.querySelector('#orlen_pickup_pickup_date') || 
                    form.querySelector('[name="orlen_pickup[pickup_date]"]');
  const hourSelect = form.querySelector('#orlen_pickup_pickup_hours') ||
                    form.querySelector('[name="orlen_pickup[pickup_hours]"]');

  if (dateSelect) {
    dateSelect.innerHTML = '<option value="">Wybierz datę</option>';
    dateSelect.disabled = true;
  }

  if (hourSelect) {
    hourSelect.innerHTML = '<option value="">Wybierz godzinę</option>';
    hourSelect.disabled = true;
  }
};

/**
 * Handle form submission
 * @param {HTMLFormElement} form - The form to submit
 * @param {string} modalId - The ID of the modal
 */
/**
 * Validate required fields in the form
 * @param {HTMLFormElement} form - The form to validate
 * @returns {boolean} - Whether the form is valid
 */
const validateRequiredFields = (form) => {
  let isValid = true;
  
  // Check all elements with required attribute
  const requiredInputs = form.querySelectorAll('[required]');
  requiredInputs.forEach(input => {
    if (!input.value.trim()) {
      showErrorMessage(input, 'To pole jest wymagane');
      isValid = false;
    }
  });
  
  // Check select elements with required attribute
  const requiredSelects = form.querySelectorAll('select[required]');
  requiredSelects.forEach(select => {
    if (!select.value || select.value === '') {
      showErrorMessage(select, 'To pole jest wymagane');
      isValid = false;
    }
  });
  
  // Check for elements that might be required based on their name
  const pickupDateSelect = form.querySelector('#orlen_pickup_pickup_date') || 
                            form.querySelector('[name="orlen_pickup[pickup_date]"]');
  const pickupHoursSelect = form.querySelector('#orlen_pickup_pickup_hours') || 
                             form.querySelector('[name="orlen_pickup[pickup_hours]"]');
  const postCodeInput = form.querySelector('#orlen_pickup_PostCode') || 
                         form.querySelector('[name="orlen_pickup[PostCode]"]');
                         
  // Check required fields by name (especially for dynamically loaded fields)
  if (postCodeInput && !postCodeInput.value.trim()) {
    showErrorMessage(postCodeInput, 'Kod pocztowy jest wymagany');
    isValid = false;
  }
  
  if (pickupDateSelect && !pickupDateSelect.disabled && (!pickupDateSelect.value || pickupDateSelect.value === '')) {
    showErrorMessage(pickupDateSelect, 'Data odbioru jest wymagana');
    isValid = false;
  }
  
  if (pickupHoursSelect && !pickupHoursSelect.disabled && (!pickupHoursSelect.value || pickupHoursSelect.value === '')) {
    showErrorMessage(pickupHoursSelect, 'Godzina odbioru jest wymagana');
    isValid = false;
  }
  
  return isValid;
};

export const setupAjaxSubmission = (modalId, form) => {
  if (!form) return;

  const submitBtn = document.querySelector(`#${modalId} .js-submit-modal-form-orlenpaczka-btn`);
  if (!submitBtn) return;

  submitBtn.addEventListener('click', async (event) => {
    try {
      event.preventDefault();
      hideAllErrors(form);
      
      // Validate required fields
      if (!validateRequiredFields(form)) {
        return; // Stop submission if validation fails
      }

      // Remove existing error messages
      const genericErrors = form.querySelectorAll('.alert.alert-danger');
      genericErrors.forEach(errorDiv => errorDiv.remove());

      // Remove old package inputs
      const oldPackageInputs = form.querySelectorAll('[name="orlen_pickup[packageIds][]"]');
      oldPackageInputs.forEach(input => input.remove());

      // Handle bulk package selection
      const $form = jQuery(form);
      const $bulkInputsBlock = jQuery('#orlen_pickup_packageIds');

      if ($bulkInputsBlock.length > 0) {
        let $checkboxes = [];

        if (window.orlenpaczkaGrid) {
          try {
            const grid = window.orlenpaczkaGrid;
            const GridMap = {
              bulks: {
                checkedCheckbox: '.js-bulk-action-checkbox:checked'
              }
            };

            $checkboxes = grid
              .getContainer()
              .find(GridMap.bulks.checkedCheckbox);
          } catch (gridError) {
            $checkboxes = jQuery('.js-bulk-action-checkbox:checked');
          }
        } else {
          $checkboxes = jQuery('.js-bulk-action-checkbox:checked');
        }

        if ($bulkInputsBlock.length > 0 && $checkboxes.length > 0) {
          $checkboxes.each(function() {
            const $checkbox = jQuery(this);
            const checkboxValue = $checkbox.val();
            const prototype = $bulkInputsBlock.data('prototype').replace(/__name__/g, checkboxValue);
            const $input = jQuery(prototype);
            $input.val(checkboxValue);
            $form.append($input);
          });
        }
      }

      const formData = new FormData(form);
      
      // Disable submit button and show loading state
      submitBtn.disabled = true;
      const originalText = submitBtn.innerHTML;
      submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Wysyłanie...';

      const data = await sendFormData(form.action, formData);
      
      // Re-enable submit button
      submitBtn.disabled = false;
      submitBtn.innerHTML = originalText;

      if (data.success === true) {
        if (data.redirect) {
          window.location.href = data.redirect;
        } else {
          alert('Operacja zakończona sukcesem!');
          const modal = document.querySelector('#' + modalId);
          if (modal && typeof bootstrap !== 'undefined' && bootstrap.Modal) {
            const modalInstance = bootstrap.Modal.getInstance(modal);
            if (modalInstance) modalInstance.hide();
          }
        }
      } else {
        hideAllErrors(form);

        if (data.errors) {
          Object.keys(data.errors).forEach(fieldName => {
            const errorMessage = data.errors[fieldName];
            const field = form.querySelector(`#orlen_pickup_${fieldName}`) ||
                          form.querySelector(`[name="orlen_pickup[${fieldName}]"]`);

            if (field) {
              showErrorMessage(field, errorMessage);
            } else {
              form.prepend(createGeneralErrorDiv(errorMessage));
            }
          });
        } else if (data.message) {
          form.prepend(createGeneralErrorDiv(data.message));
        } else {
          alert('Wystąpił błąd podczas przetwarzania formularza.');
        }
      }
    } catch (e) {
      submitBtn.disabled = false;
      submitBtn.innerHTML = originalText;
      alert('Wystąpił błąd podczas przetwarzania formularza');
      console.error(e);
    }
  });
};
