/**
 * Pickup Form Component for Orlen Paczka module
 */
import { showErrorMessage, hideErrorMessage, hideAllErrors, markRequiredFields } from '../utils/form-utils.js';
import { fetchAvailableDates, fetchAvailableHours } from '../services/api-service.js';
import { resetFormValues, setupAjaxSubmission } from '../handlers/form-handler.js';

/**
 * Form state for postal code tracking
 */
const formState = {
  currentPostalCode: '',
  requestSent: false
};

/**
 * Reset form state when needed
 */
export const resetFormState = () => {
  formState.currentPostalCode = '';
  formState.requestSent = false;
};

/**
 * Check if the postal code request should be sent
 * @param {string} postCode - The postal code to check
 * @returns {boolean} - Whether the request should be sent
 */
const shouldSendRequest = (postCode) => {
  if (!postCode || postCode.length < 5) return false;
  // If the form was just reset, we should allow a new request even for the same postal code
  if (postCode === formState.currentPostalCode && formState.requestSent && formState.currentPostalCode !== '') return false;
  return true;
};

/**
 * Handle the postal code submission
 * @param {string} postCode - The postal code
 * @param {HTMLSelectElement} dateSelect - The date select element
 * @param {HTMLSelectElement} hourSelect - The hour select element
 * @param {string} dateUrl - The URL to fetch dates from
 */
const handlePostCodeSubmit = (postCode, dateSelect, hourSelect, form) => {
  if (postCode && postCode.length >= 5) {
    resetSelectElements(dateSelect, hourSelect);
    fetchDatesForPostCode(postCode, dateSelect, form);
  }
};

/**
 * Reset the select elements
 * @param {HTMLSelectElement} dateSelect - The date select element
 * @param {HTMLSelectElement} hourSelect - The hour select element
 */
const resetSelectElements = (dateSelect, hourSelect) => {
  dateSelect.disabled = true;
  dateSelect.innerHTML = '<option value="">Wybierz datę</option>';
  
  if (hourSelect) {
    hourSelect.disabled = true;
    hourSelect.innerHTML = '<option value="">Wybierz godzinę</option>';
  }
};

/**
 * Fetch dates for a postal code
 * @param {string} postCode - The postal code
 * @param {HTMLSelectElement} dateSelect - The date select element
 * @param {HTMLFormElement} form - The form element
 */
const fetchDatesForPostCode = async (postCode, dateSelect, form) => {
  hideAllErrors(form);

  const dateUrl = form.getAttribute('data-date-url');
  if (!dateUrl) return;

  try {
    const data = await fetchAvailableDates(dateUrl, postCode);
    
    if (!data.success) {
      const postCodeInput = form.querySelector('#orlen_pickup_PostCode') || 
                            form.querySelector('[name="orlen_pickup[PostCode]"]');
      if (postCodeInput) {
        showErrorMessage(postCodeInput, data.message || 'Błąd pobierania dostępnych dat');
      }
      return;
    }

    if (data.dates && Array.isArray(data.dates)) {
      data.dates.forEach(date => {
        const option = document.createElement('option');
        option.value = date.value;
        option.textContent = date.label;
        dateSelect.appendChild(option);
      });

      if (data.dates.length > 0) {
        dateSelect.disabled = false;
      }
    }
  } catch (error) {
    console.error('Error fetching dates:', error);
  }
};

/**
 * Fetch available hours for a selected date
 * @param {string} date - The selected date
 * @param {HTMLSelectElement} hourSelect - The hour select element
 * @param {HTMLFormElement} form - The form element
 */
const fetchHoursForDate = async (date, hourSelect, form) => {
  if (!hourSelect) return;
  
  hideAllErrors(form);
  
  const dateSelect = form.querySelector('#orlen_pickup_pickup_date') || 
                     form.querySelector('[name="orlen_pickup[pickup_date]"]');
  
  hideErrorMessage(dateSelect);
  
  hourSelect.disabled = true;
  hourSelect.innerHTML = '<option value="">Wybierz godzinę</option>';
  
  const hourUrl = form.getAttribute('data-hour-url');
  if (!hourUrl) return;
  
  const postData = { date };
  const postCodeInput = form.querySelector('#orlen_pickup_PostCode') || 
                       form.querySelector('[name="orlen_pickup[PostCode]"]');
  
  if (postCodeInput) {
    postData.postCode = postCodeInput.value;
  }
  
  try {
    const data = await fetchAvailableHours(hourUrl, postData);
    
    if (!data.success) {
      showErrorMessage(dateSelect, data.message || 'Błąd pobierania dostępnych godzin');
      return;
    }
    
    if (data.hours) {
      const availableHours = data.hours.from || [];
      
      availableHours.forEach(hour => {
        const option = document.createElement('option');
        option.value = hour.value;
        option.textContent = hour.label;
        hourSelect.appendChild(option);
      });
      
      if (availableHours.length > 0) {
        hourSelect.disabled = false;
      }
    }
  } catch (error) {
    console.error('Error fetching hours:', error);
  }
};

/**
 * Setup postal code input events
 * @param {HTMLInputElement} postCodeInput - The postal code input
 * @param {HTMLSelectElement} dateSelect - The date select element
 * @param {HTMLSelectElement} hourSelect - The hour select element
 * @param {HTMLFormElement} form - The form element
 */
const setupPostCodeEvents = (postCodeInput, dateSelect, hourSelect, form) => {
  postCodeInput.addEventListener('input', () => {
    hideErrorMessage(postCodeInput);
    const postCode = postCodeInput.value.trim();
    
    if (postCode !== formState.currentPostalCode) {
      formState.requestSent = false;
    }
    
    if (postCode.length === 6 && shouldSendRequest(postCode)) {
      formState.currentPostalCode = postCode;
      formState.requestSent = true;
      handlePostCodeSubmit(postCode, dateSelect, hourSelect, form);
    }
  });
  
  postCodeInput.addEventListener('blur', () => {
    const postCode = postCodeInput.value.trim();
    
    if (postCode.length >= 5 && shouldSendRequest(postCode)) {
      formState.currentPostalCode = postCode;
      formState.requestSent = true;
      handlePostCodeSubmit(postCode, dateSelect, hourSelect, form);
    }
  });
};

/**
 * Setup date select change event
 * @param {HTMLSelectElement} dateSelect - The date select element
 * @param {HTMLSelectElement} hourSelect - The hour select element
 * @param {HTMLFormElement} form - The form element
 */
const setupDateSelectEvent = (dateSelect, hourSelect, form) => {
  dateSelect.addEventListener('change', () => {
    hideErrorMessage(dateSelect);
    const selectedDate = dateSelect.value;
    
    if (selectedDate && hourSelect) {
      fetchHoursForDate(selectedDate, hourSelect, form);
    } else if (hourSelect) {
      hourSelect.disabled = true;
      hourSelect.innerHTML = '<option value="">Wybierz godzinę</option>';
    }
  });
};

/**
 * Initialize the pickup form
 */
export const initPickupForm = () => {
  const pickupForm = document.querySelector('#callPickupModal form');
  const callPickupModal = document.querySelector('#callPickupModal');
  
  if (!pickupForm) return;
  
  const resetBeforeShow = () => {
    resetFormValues(pickupForm);
    resetFormState();
    markRequiredFields(pickupForm);
  };
  
  // Setup jQuery modal events (for backward compatibility)
  if (typeof jQuery !== 'undefined' && jQuery.fn.modal) {
    jQuery(callPickupModal)
      .on('show.bs.modal', resetBeforeShow)
      .on('hidden.bs.modal', () => {
        resetFormValues(pickupForm);
        resetFormState();
      });
  }
  
  // Setup native modal events
  if (callPickupModal) {
    callPickupModal.addEventListener('show.bs.modal', resetBeforeShow);
    callPickupModal.addEventListener('hidden.bs.modal', () => {
      resetFormValues(pickupForm);
      resetFormState();
    });
  }
  
  // Handle modal triggers via data attributes
  document.addEventListener('click', (event) => {
    const target = event.target;
    
    if ((target.hasAttribute('data-toggle') || target.hasAttribute('data-bs-toggle')) &&
        (target.getAttribute('data-target') === '#callPickupModal' ||
         target.getAttribute('data-bs-target') === '#callPickupModal')) {
      resetBeforeShow();
    }
  });
  
  // Handle modal close buttons
  document.addEventListener('click', (event) => {
    const target = event.target;
    
    if ((target.hasAttribute('data-dismiss') && target.getAttribute('data-dismiss') === 'modal') ||
        (target.hasAttribute('data-bs-dismiss') && target.getAttribute('data-bs-dismiss') === 'modal') ||
        target.classList.contains('close') ||
        target.classList.contains('btn-close')) {
      
      if (target.closest('#callPickupModal')) {
        resetFormValues(pickupForm);
        resetFormState();
      }
    }
  });
  
  resetFormValues(pickupForm);
  setupAjaxSubmission('callPickupModal', pickupForm);
  hideAllErrors(pickupForm);
  
  // Mark required fields with asterisks
  markRequiredFields(pickupForm);
  
  // Get form elements
  const postCodeInput = pickupForm.querySelector('#orlen_pickup_PostCode') || 
                       pickupForm.querySelector('[name="orlen_pickup[PostCode]"]');
  const dateSelect = pickupForm.querySelector('#orlen_pickup_pickup_date') || 
                     pickupForm.querySelector('[name="orlen_pickup[pickup_date]"]');
  const hourSelect = pickupForm.querySelector('#orlen_pickup_pickup_hours') || 
                    pickupForm.querySelector('[name="orlen_pickup[pickup_hours]"]');
  
  if (!dateSelect) return;
  
  // Setup initial form state
  dateSelect.disabled = true;
  if (hourSelect) hourSelect.disabled = true;
  
  // Setup events
  if (postCodeInput) {
    setupPostCodeEvents(postCodeInput, dateSelect, hourSelect, pickupForm);
  }
  
  setupDateSelectEvent(dateSelect, hourSelect, pickupForm);
};
