/**
 * Custom Ajax Bulk Action Extension that supports direct URLs
 */
class CustomAjaxBulkActionExtension {
  constructor() {
    // We don't initialize the router since we'll be using direct URLs
    this.abortController = null;
  }

  extend(grid) {
    grid
      .getContainer()
      .on('click', '.js-bulk-action-ajax-btn', (event) => {
        const $ajaxButton = $(event.currentTarget);
        const $checkboxes = $('input[type="checkbox"][name="bulk_action_selected_checkbox[]"]:checked');
        const selectedIds = $checkboxes.get().map((checkbox) => checkbox.value);

        if (selectedIds.length === 0) {
          return;
        }

        const confirmBulkAction = $ajaxButton.data('confirmBulkAction') ?? true;

        if (confirmBulkAction) {
          const progressionTitle = $ajaxButton.data('progressTitle');
          const closeButtonLabel = $ajaxButton.data('cancelLabel') || 'Cancel';
          const confirmTitle = $ajaxButton.data('confirmTitle') || 'Apply modifications';
          const bulkAction = $ajaxButton.data('bulkAction') ?? 'bulk-action';

          // Create confirm modal
          const confirmModal = new window.prestashop.component.ConfirmModal({
            id: `${grid.id}-ajax-${bulkAction}_ajax-confirm-modal`,
            modalTitle: confirmTitle,
            closeButtonLabel,
            confirmMessage: progressionTitle.replace('%total%', selectedIds.length),
            confirmButtonLabel: confirmTitle,
            confirmCallback: () => {
              this.submitForm(grid, $(event.currentTarget), selectedIds);
            },
          });
          confirmModal.show();
        } else {
          this.submitForm(grid, $(event.currentTarget), selectedIds);
        }
      });
  }

  async submitForm(grid, $ajaxButton, selectedIds) {
    const bulkChunkSize = $ajaxButton.data('requestBulkChunkSize') ?? 10;
    const reloadAfterBulk = $ajaxButton.data('reloadAfterBulk') ?? true;

    const bulkAction = $ajaxButton.data('bulkAction') ?? 'bulk-action';
    const progressionTitle = $ajaxButton.data('progressTitle');
    const progressionMessage = $ajaxButton.data('progressMessage');
    const closeLabel = $ajaxButton.data('closeLabel');
    const abortProcessingLabel = $ajaxButton.data('stopProcessing');
    const errorsMessage = $ajaxButton.data('errorsMessage');
    const backToProcessingLabel = $ajaxButton.data('backToProcessing');
    const downloadErrorLogLabel = $ajaxButton.data('downloadErrorLog');
    const viewErrorLogLabel = $ajaxButton.data('viewErrorLog');
    const viewErrorTitle = $ajaxButton.data('viewErrorTitle');

    this.abortController = new AbortController();

    // Create a new progress modal
    const modal = new window.prestashop.component.ProgressModal({
      id: `${grid.id}-ajax-${bulkAction}_ajax-progress-modal`,
      abortCallback: () => {
        this.abortController.abort();
      },
      closeCallback: () => {
        if (reloadAfterBulk) {
          window.location.reload();
        }
      },
      progressTitle: progressionTitle.replace('%total%', selectedIds.length),
      progressMessage: progressionMessage.replace('%done%', 0).replace('%total%', selectedIds.length),
      closeLabel,
      abortProcessingLabel,
      errorsMessage,
      backToProcessingLabel,
      downloadErrorLogLabel,
      viewErrorLogLabel,
      viewErrorTitle,
    });

    // Show the modal
    modal.show();

    const chunkedIds = [];
    for (let i = 0; i < selectedIds.length; i += bulkChunkSize) {
      chunkedIds.push(selectedIds.slice(i, i + bulkChunkSize));
    }

    let doneCount = 0;

    // Process each chunk
    for (const chunkIds of chunkedIds) {
      try {
        // This is the key difference - we call our custom method
        const response = await this.callAjaxAction($ajaxButton, chunkIds, this.abortController.signal);

        if (this.abortController.signal.aborted) {
          break;
        }

        doneCount += chunkIds.length;
        modal.updateProgress(doneCount, selectedIds.length);

        // Process the response
        if (!response.ok) {
          const data = await response.json();
          if (Array.isArray(data.errors) && data.errors.length > 0) {
            data.errors.forEach((error) => {
              modal.addError(error);
            });
          } else {
            modal.addError(data.errors ?? data.error ?? data.message);
          }
        }
      } catch (error) {
        modal.addError(error.message || 'An error occurred');
      }
    }

    modal.completeProgress();
  }

  // This is the overridden method that uses direct URL instead of router.generate
  callAjaxAction($ajaxButton, chunkIds, abortSignal) {
    const requestParamName = $ajaxButton.data('requestParamName') ?? 'bulk_ids';
    const routeParams = $ajaxButton.data('routeParams') ?? {};
    const routeMethod = $ajaxButton.data('routeMethod') ?? 'POST';

    const formData = new FormData();
    chunkIds.forEach((chunkId, index) => {
      formData.append(`${requestParamName}[${index}]`, chunkId);
    });

    let requestMethod;

    switch (routeMethod.toUpperCase()) {
      case 'PATCH':
      case 'DELETE':
        requestMethod = 'POST';
        break;
      default:
        requestMethod = routeMethod;
        break;
    }

    // Use direct URL from data-ajax-url instead of router.generate
    // This is the key change from the original method
    const url = $ajaxButton.data('ajax-url');

    // Add any route params as query string to the URL
    const urlObj = new URL(url, window.location.origin);
    Object.entries(routeParams).forEach(([key, value]) => {
      urlObj.searchParams.append(key, value);
    });

    return fetch(urlObj.toString(), {
      method: requestMethod,
      body: formData,
      headers: {
        _method: routeMethod,
      },
      signal: abortSignal,
    });
  }
}
