<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Soap\ExtSoapEngine;

use Orlen\OrlenPaczka\Soap\Engine\DecoderInterface;
use Orlen\OrlenPaczka\Soap\Engine\DriverInterface;
use Orlen\OrlenPaczka\Soap\Engine\EncoderInterface;
use Orlen\OrlenPaczka\Soap\Engine\HttpBinding\SoapRequest;
use Orlen\OrlenPaczka\Soap\Engine\HttpBinding\SoapResponse;
use Orlen\OrlenPaczka\Soap\Engine\Metadata\LazyInMemoryMetadata;
use Orlen\OrlenPaczka\Soap\Engine\Metadata\MetadataInterface;
use Orlen\OrlenPaczka\Soap\ExtSoapEngine\Generator\DummyMethodArgumentsGenerator;

final class ExtSoapDriver implements DriverInterface
{
    /**
     * @var AbusedClient
     */
    private $client;
    /**
     * @var EncoderInterface
     */
    private $encoder;
    /**
     * @var DecoderInterface
     */
    private $decoder;
    /**
     * @var MetadataInterface
     */
    private $metadata;

    public function __construct(
        AbusedClient $client,
        EncoderInterface $encoder,
        DecoderInterface $decoder,
        MetadataInterface $metadata
    ) {
        $this->client = $client;
        $this->encoder = $encoder;
        $this->decoder = $decoder;
        $this->metadata = $metadata;
    }

    public static function createFromOptions(ExtSoapOptions $options): self
    {
        $client = AbusedClient::createFromOptions($options);

        return self::createFromClient(
            $client,
            new LazyInMemoryMetadata(new ExtSoapMetadata($client))
        );
    }

    public static function createFromClient(AbusedClient $client, ?MetadataInterface $metadata = null): self
    {
        $metadata = $metadata ?? new LazyInMemoryMetadata(new ExtSoapMetadata($client));

        return new self(
            $client,
            new ExtSoapEncoder($client),
            new ExtSoapDecoder($client, new DummyMethodArgumentsGenerator($metadata)),
            $metadata
        );
    }

    public function decode(string $method, SoapResponse $response)
    {
        return $this->decoder->decode($method, $response);
    }

    public function encode(string $method, array $arguments): SoapRequest
    {
        return $this->encoder->encode($method, $arguments);
    }

    public function getMetadata(): MetadataInterface
    {
        return $this->metadata;
    }

    public function getClient(): AbusedClient
    {
        return $this->client;
    }
}
