<?php

namespace Orlen\OrlenPaczka\Hook;

use Orlen\OrlenPaczka\Hook\Interfaces\HookInterface;
use Orlen\OrlenPaczka\Repository\PackageRepository;
use Orlen\OrlenPaczka\Translator\Adapter\Translator;

class ActionOrderStatusUpdate implements HookInterface
{
    private PackageRepository $packageRepository;
    private Translator $translator;

    public function __construct(PackageRepository $packageRepository, Translator $translator)
    {
        $this->packageRepository = $packageRepository;
        $this->translator = $translator;
    }

    public function execute($params)
    {
        // Sprawdzamy czy mamy obiekt statusu zamówienia
        if (!isset($params['newOrderStatus'])) {
            return;
        }

        $newOrderStatus = $params['newOrderStatus'];

        // Sprawdzamy czy status jest typu "delivery" (wysłane) i shipped = 0
        if ($newOrderStatus->delivery != 1 || $newOrderStatus->shipped != 1 || $newOrderStatus->send_email != 1) {
            return;
        }

        // Pobieramy numery paczek dla tego zamówienia
        $packageCodes = $this->packageRepository->getPackagesFromOrderShipped($params['id_order']);

        // Jeśli nie ma żadnych paczek, kończymy
        if (empty($packageCodes)) {
            return;
        }

        // Pobieramy zamówienie
        $order = new \Order((int)$params['id_order']);
        $customer = new \Customer((int)$order->id_customer);

        // Przygotowujemy linki do śledzenia
        $trackingLinks = [];
        foreach ($packageCodes as $code) {
            $trackingLinks[] = [
                'code' => $code,
                'url' => 'https://www.orlenpaczka.pl/sledz-paczke/?numer=' . $code
            ];
        }

        // Przygotowujemy dane do szablonu maila
        $templateVars = [
            '{firstname}' => $customer->firstname,
            '{lastname}' => $customer->lastname,
            '{order_name}' => $order->getUniqReference(),
            '{tracking_links}' => $this->formatTrackingLinks($trackingLinks),
            '{tracking_numbers}' => implode(', ', $packageCodes),
        ];

        // Wysyłamy maila
        \Mail::Send(
            (int)$order->id_lang,
            'orlenpaczka_shipping',
            $this->translator->trans('Your shipment has been sent - ORLEN Paczka', [], 'Modules.Orlenpaczka.Notification'),
            $templateVars,
            $customer->email,
            $customer->firstname . ' ' . $customer->lastname,
            null,
            null,
            null,
            null,
            _PS_MODULE_DIR_ . 'orlenpaczka/mails/',
            false,
            (int)$order->id_shop
        );
    }

    /**
     * Formatuje linki do śledzenia dla szablonu email
     */
    private function formatTrackingLinks(array $trackingLinks): string
    {
        $html = '';
        foreach ($trackingLinks as $link) {
            $html .= sprintf(
                '<div class="tracking-box"><strong>Numer przesyłki:</strong> %s<br><a href="%s" target="_blank">Śledź przesyłkę</a></div>',
                $link['code'],
                $link['url']
            );
        }
        return $html;
    }
}
