<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Controller\Admin;

use Orlen\OrlenPaczka\Api\OrlenPaczka\OrlenPaczkaClientFactory;
use Orlen\OrlenPaczka\Api\OrlenPaczka\Type\GivePartnerStatus;
use Orlen\OrlenPaczka\Api\OrlenPaczka\Type\Ping;
use Orlen\OrlenPaczka\Authorization\Dto\CredentialsInterface;
use Orlen\OrlenPaczka\Authorization\Provider\CredentialsProviderInterface;
use Orlen\OrlenPaczka\Form\Handler\AuthorizationHandler;
use Orlen\OrlenPaczka\Form\Provider\AuthorizationProvider;
use Orlen\OrlenPaczka\Form\Type\AuthorizationType;
use Orlen\OrlenPaczka\Tabs\Tabs;
use Orlen\OrlenPaczka\Translator\Adapter\Translator;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route(path="/config", name="orlenpaczka.config.authorization")
 */
class AuthorizationController extends FrameworkBundleAdminController
{
    const TAB_NAME = 'AdminModules';
    private Translator $translator;

    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
    }

    /**
     * @Route("/authorization", name=".index", methods={"GET", "POST"})
     */
    public function authorizationAction(
        \Module $module,
        Request $request,
        Tabs $tabs,
        AuthorizationProvider $authorizationProvider,
        AuthorizationHandler $authorizationHandler
    ): Response
    {
        $form = $this->createForm(AuthorizationType::class, $authorizationProvider->getData());

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $authorizationHandler->handle($form->getData());
        }

        return $this->render('@Modules/orlenpaczka/views/templates/admin/config/tabs/credentials.html.twig', [
            'layoutTitle' => $module->displayName,
            'help_link' => false,
            'tabs' => $tabs,
            'form' => $form->createView(),
            'title' => $this->translator->trans('Authorization', [], 'Modules.Orlenpaczka.Tab'),
        ]);
    }
    /**
     * @Route("/update-sandbox", name=".update_sandbox", methods={"POST"})
     */
    public function updateSandboxAction(Request $request, CredentialsProviderInterface $credentialsProvider)
    {
        $sandbox = (bool)$request->get('sandbox');
        $credentials = $credentialsProvider->getCredentials($sandbox);
        if (!$credentials instanceof CredentialsInterface) {
            return new JsonResponse(['partner_id' => '', 'partner_key' => '', 'success' => true]);
        }
        return new JsonResponse(['partner_id' => $credentials->getPartnerId(), 'partner_key' => $credentials->getPartnerKey(), 'success' => true]);
    }
    /**
     * @Route("/test-connection", name=".test_connection", methods={"POST", "GET"})
     */

    public function testConnectionAction(OrlenPaczkaClientFactory $factory)
    {
        try {
            $client = $factory->factory();
            $sandbox = $client->isSandbox()?
                $this->translator->trans('sandbox', [], 'Modules.Orlenpaczka.Notification')
                : $this->translator->trans('production', [], 'Modules.Orlenpaczka.Notification');
            $credentials = $client->getCredentials();
            $response = $client->givePartnerStatus(new GivePartnerStatus($credentials->getPartnerId(), $credentials->getPartnerKey()));
        }catch (\Exception|\Throwable $e){
            return new JsonResponse(['success' => false, 'error' => $this->translator->trans('Error connection', [], 'Modules.Orlenpaczka.Notification')]);
        }
        $partnerStatusResult = $response->getGivePartnerStatusResult()->geXml(false);

        return $partnerStatusResult->NewDataSet[0]->GivePartnerStatus[0]->Err == '000'?
            new JsonResponse(['success' => true, 'message' => $this->translator->trans('Credentials (%s , %s) in mode %s are correct', [$credentials->getPartnerId(), $credentials->getPartnerKey(), $sandbox], 'Modules.Orlenpaczka.Notification')]):
            new JsonResponse(['success' => false, 'error' => $this->translator->trans('Credentials (%s , %s) in mode %s are incorrect', [$credentials->getPartnerId(), $credentials->getPartnerKey(), $sandbox], 'Modules.Orlenpaczka.Notification')]);
    }
    /**
     * @Route("/ping", name=".ping", methods={"POST", "GET"})
     */

    public function pingAction(OrlenPaczkaClientFactory $factory)
    {
        try{
            $client = $factory->factory();
            $response = $client->ping(new Ping());
        }catch (\Exception|\Throwable $e){
            return new JsonResponse(['success' => false, 'error' => $this->translator->trans('The service is unavailable', [], 'Modules.Orlenpaczka.Notification')]);
        }
        return $response->getPingResult()?
            new JsonResponse(['success' => true, 'message' => $this->translator->trans('The service is active', [], 'Modules.Orlenpaczka.Notification')]):
            new JsonResponse(['success' => false, 'error' => $this->translator->trans('The service is unavailable', [], 'Modules.Orlenpaczka.Notification')]);


    }
}
