<?php

namespace Orlen\OrlenPaczka\Repository;

use Orlen\OrlenPaczka\Entity\Address;
use Orlen\OrlenPaczka\Entity\Package;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\OptimisticLockException;
use Doctrine\ORM\ORMException;
use Doctrine\Persistence\ManagerRegistry;
use Orlen\OrlenPaczka\Entity\PackageSelectedPoint;
use Orlen\OrlenPaczka\Entity\SelectedPoint;
use Orlen\OrlenPaczka\Entity\Shipment;
use Orlen\OrlenPaczka\Entity\ShipmentType;

/**
 * @extends ServiceEntityRepository<Package>
 *
 * @method Package|null find($id, $lockMode = null, $lockVersion = null)
 * @method Package|null findOneBy(array $criteria, array $orderBy = null)
 * @method Package[]    findAll()
 * @method Package[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class PackageRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Package::class);
    }

    /**
     * @throws ORMException
     * @throws OptimisticLockException
     */
    public function add(Package $entity, bool $flush = true): void
    {
        $this->_em->persist($entity);
        if ($flush) {
            $this->_em->flush();
        }
    }

    public function flush()
    {
        $this->_em->flush();
    }

    /**
     * @throws ORMException
     * @throws OptimisticLockException
     */
    public function remove(Package $entity, bool $flush = true): void
    {
        $this->_em->remove($entity);
        if ($flush) {
            $this->_em->flush();
        }
    }

    public function getDataLabel($ids, $allowRegenerate = false)
    {
        $qb =$this->_em->getConnection()->createQueryBuilder();
        $qb
            ->select(
                '
                p.id_package,
                sp.destination_code as DestinationCode,
                s.size as BoxSize,
                c.email as EMail,
                c.firstname as FirstName,
                c.lastname as LastName,
                a.company as CompanyName,
                IF(a.phone IS NULL OR a.phone = "", a.phone_mobile, a.phone) as PhoneNumber,
                a.city as City,
                a.postcode as PostCode,
                sa.email as SenderEMail,
                sa.first_name as SenderFirstName,
                sa.last_name as SenderLastName,
                sa.street as SenderStreetName,
                sa.building_number as SenderBuildingNumber,
                sa.city as SenderCity,
                sa.postal_code as SenderPostCode,
                sa.phone as SenderPhoneNumber,
                CASE s.sender_orders_type
                    WHEN 0 THEN s.sender_orders
                    WHEN 1 THEN o.reference
                    WHEN 2 THEN CAST(o.id_order AS CHAR)
                    WHEN 3 THEN NULL
                    ELSE s.sender_orders
                END as SenderOrders
                '
            )
            ->from(Package::TABLE_NAME, 'p')
            ->innerJoin('p', _DB_PREFIX_.'orders', 'o', 'p.id_order = o.id_order')
            ->innerJoin('o', _DB_PREFIX_.'customer', 'c', 'o.id_customer = c.id_customer')
            ->innerJoin('o', _DB_PREFIX_.'address', 'a', 'o.id_address_delivery = a.id_address')
            ->innerJoin('p', Shipment::TABLE_NAME, 's', 'p.id_shipment = s.id_shipment')
            ->innerJoin('p', PackageSelectedPoint::TABLE_NAME, 'sp', 'sp.id_pickup = p.id_pickup')
            ->innerJoin('s', ShipmentType::TABLE_NAME, 'st', 's.id_shipment_type = st.id_shipment_type')
            ->innerJoin('p', Address::TABLE_NAME, 'sa', 'sa.id_address = 1')

        ->where($qb->expr()->in('p.id_package', ':ids'))
        ->andWhere('p.id_delivery IS NULL');
        if (!$allowRegenerate){
            $qb->andWhere('p.pack_code IS NULL');
        }


        $qb->setParameter('ids', $ids, \Doctrine\DBAL\Connection::PARAM_INT_ARRAY);

        // Execute query i zwróć wyniki jako tablicę asocjacyjną
        return $qb->execute()->fetchAll();
    }

    public function getPackagesFromOrderToGenerateLabel($idOrders)
    {
        $qb = $this->createQueryBuilder('p');
        $qb
            ->select('p.id')
            ->andWhere('p.idOrder IN (:ids)')
            ->setParameter('ids', $idOrders, \Doctrine\DBAL\Connection::PARAM_INT_ARRAY);

        // Pobierz wyniki jako tablicę asocjacyjną
        $result = $qb->getQuery()->getArrayResult();

        // Przekształć tablicę asocjacyjną na płaską tablicę wartości ID
        return array_column($result, 'id');
    }

    public function getPackagesFromOrderShipped($idOrder)
    {
        $qb = $this->createQueryBuilder('p');
        $qb
            ->select('p.packCode')
            ->andWhere('p.idOrder = :id')
            ->andWhere('p.packCode IS NOT NULL')
            ->setParameter('id', $idOrder);

        // Pobierz wyniki jako tablicę asocjacyjną
        $result = $qb->getQuery()->getArrayResult();

        // Przekształć tablicę asocjacyjną na płaską tablicę wartości ID
        return array_column($result, 'packCode');
    }


    public function canGenerateProtocol($idsPackages)
    {
        $qb = $this->createQueryBuilder('p');
        $qb
            ->andWhere('p.id IN (:ids)')
            ->andWhere($qb->expr()->orX('p.protocol IS NOT NULL', 'p.packCode IS NULL'))
            ->setParameter('ids', $idsPackages, \Doctrine\DBAL\Connection::PARAM_INT_ARRAY);
         $result = $qb->getQuery()->execute();
        return $result === null;
    }

    public function duplicateForOrder($id)
    {
        $package = $this->find($id);
        if (!$package) {
            return false;
        }
        $newPackage = new Package();
        $newPackage->setIdOrder($package->getIdOrder());
        $newPackage->setPickup($package->getPickup());
        $newPackage->setShipment($package->getShipment());
        $newPackage->setSize($package->getSize());
        $this->_em->persist($newPackage);
        $this->_em->flush();
        return true;

    }

    // /**
    //  * @return Package[] Returns an array of Package objects
    //  */
    /*
    public function findByExampleField($value)
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.exampleField = :val')
            ->setParameter('val', $value)
            ->orderBy('p.id', 'ASC')
            ->setMaxResults(10)
            ->getQuery()
            ->getResult()
        ;
    }
    */

    /*
    public function findOneBySomeField($value): ?Package
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.exampleField = :val')
            ->setParameter('val', $value)
            ->getQuery()
            ->getOneOrNullResult()
        ;
    }
    */
}
