<?php

namespace Orlen\OrlenPaczka\Protocol\Generator;

use Orlen\OrlenPaczka\Api\OrlenPaczka\OrlenPaczkaClientFactory;
use Orlen\OrlenPaczka\Api\OrlenPaczka\Type\GenerateLabelBusinessPackListTwo;
use Orlen\OrlenPaczka\Api\OrlenPaczka\Type\GenerateProtocol;
use Orlen\OrlenPaczka\Configuration\ConfigurationModule;
use Orlen\OrlenPaczka\Entity\Delivery;
use Orlen\OrlenPaczka\Entity\Protocol;
use Orlen\OrlenPaczka\Label\Settings\GenerateLabelSettings;
use Orlen\OrlenPaczka\Protocol\Interfaces\PackageProviderInterface;
use Orlen\OrlenPaczka\Protocol\Interfaces\PackageProviderItnerface;
use Orlen\OrlenPaczka\Protocol\Settings\GenerateProtocolSettings;
use Orlen\OrlenPaczka\Repository\PackageRepository;
use Orlen\OrlenPaczka\Repository\ProtocolRepository;

class ProtocolGenerator
{
    private PackageProviderInterface $packageProvider;
    private $client;
    private OrlenPaczkaClientFactory $factory;
    private PackageRepository $packageRepository;
    private ProtocolRepository $protocolRepository;

    private $result = [];
    private $translator;

    public function __construct(
        OrlenPaczkaClientFactory $factory,
        PackageProviderInterface $packageProvider,
        PackageRepository $packageRepository,
        ProtocolRepository $protocolRepository,
        $translator
    )
    {
        $this->packageProvider = $packageProvider;
        $this->factory = $factory;
        $this->packageRepository = $packageRepository;
        $this->protocolRepository = $protocolRepository;
        $this->translator = $translator;
    }

    public function generateProtocol(GenerateProtocolSettings $settings)
    {

        $this->_createClient();
        $response = $this->_sendRequest($this->_createRequest($settings));
        $labelName = $this->_saveLabel($response['label']);
        $this->_saveData($response['data'], $labelName, $settings);

        return true;

    }

    private function _createRequest(GenerateProtocolSettings $settings): GenerateProtocol
    {
        $packages = $this->packageProvider->getData($settings);
        if (empty($packages)) {
            throw new \Exception('Brak paczek do utworzenia etykiety');
        }

        $this->client = $this->factory->factory();

        $generateProtocol = new GenerateProtocol(
            $this->client->getCredentials()->getPartnerId(),
            $this->client->getCredentials()->getPartnerKey(),
            $packages,

        );
        return $generateProtocol;
    }

    private function _createClient()
    {
        $this->client = $this->factory->factory();
    }

    private function _sendRequest(GenerateProtocol $request)
    {
        $response = $this->client->generateProtocol($request);
        $data = $response->getGenerateProtocolResult()->geXml(true);
        return ['label' => $response->getLabelData(), 'data' => $data];
    }

    private function _saveData($data, $labelName, GenerateProtocolSettings $settings)
    {
        $protocol = new Protocol();
        foreach ($data->NewDataSet[0]->Table as $index => $responseObject) {

            if ($responseObject->status == 200){
                $this->result['success'][] = $this->translator->trans('Package %packCode% has been added to the protocol.', ['%packCode%' => $responseObject->PackCodeRUCH], 'Modules.Orlenpaczka.Protocol');
            }else{
                $this->result['errors'][] = $this->result[] = $this->translator->trans('Package %packCode% was not added to the protocol because it is out for delivery or has been cancelled.', ['%packCode%' => $responseObject->PackCodeRUCH], 'Modules.Orlenpaczka.Protocol');
                continue;
            }
            $protocol->setStatus($responseObject->status_opis);
            $protocol->setCode($responseObject->ProtocolCode);
            $protocol->setLabelName($labelName);
            $protocol->setLabelDateGenerated(new \DateTime());

            $package = $this->packageRepository->findoneBy(['packCode' => $responseObject->PackCodeRUCH]);
            $protocol->addPackage($package);
            if (isset($settings->getSettings()['provider']['delivery']) && $settings->getSettings()['provider']['delivery'] instanceof Delivery){
                $protocol->setDelivery($settings->getSettings()['provider']['delivery']);
            }


        }
        if ($protocol->getPackages()->count() > 0){
            $this->protocolRepository->add($protocol);
        }
    }

    /**
     * Ensure label directory exists, create if not
     *
     * @return bool
     */
    private function _ensureProtocolDirectoryExists()
    {
        if (!is_dir(ConfigurationModule::PROTOCOL_PATH)) {
            return mkdir(ConfigurationModule::PROTOCOL_PATH, 0755, true);
        }

        return true;
    }

    /**
     * Save label to file
     *
     * @param string $label
     * @param GenerateLabelSettings $settings
     * @return string
     */
    private function _saveLabel($label)
    {
        $this->_ensureProtocolDirectoryExists();

        $name = md5(uniqid(rand(), true));
        $ext =  'pdf';
        $fullName = $name.'.'.$ext;
        $filename = ConfigurationModule::PROTOCOL_PATH.$fullName;
        file_put_contents($filename, $label);

        return $fullName;
    }

    public function getSuccess()
    {
        return $this->result['success']?? [];
    }

    public function getErrors()
    {
        return $this->result['errors']?? [];
    }
}
