<?php

namespace Orlen\OrlenPaczka\Label\Generator;

use Orlen\OrlenPaczka\Api\OrlenPaczka\OrlenPaczkaClientFactory;
use Orlen\OrlenPaczka\Api\OrlenPaczka\Type\GenerateLabelBusinessPackListTwo;
use Orlen\OrlenPaczka\Configuration\ConfigurationModule;
use Orlen\OrlenPaczka\Exception\NoAddresSenderException;
use Orlen\OrlenPaczka\Label\Exception\NoPackageException;
use Orlen\OrlenPaczka\Label\Interfaces\PackageProvider;
use Orlen\OrlenPaczka\Label\Settings\GenerateLabelSettings;
use Orlen\OrlenPaczka\Repository\AddressRepository;
use Orlen\OrlenPaczka\Repository\PackageRepository;
use Orlen\OrlenPaczka\Repository\PackageSelectedPointRepository;

class LabelGenerator
{
    private OrlenPaczkaClientFactory $factory;
    private PackageProvider $packageProvider;
    private PackageRepository $packageRepository;
    private \Orlen\OrlenPaczka\Api\OrlenPaczka\OrlenPaczkaClient $client;
    private AddressRepository $addressRepository;

    private array $result = [];
    const SUCCESS_CODE = ['000', '006', '007', '008'];
    private PackageSelectedPointRepository $packageSelectedPointRepository;

    public function __construct(
        OrlenPaczkaClientFactory $factory,
        PackageProvider $packageProvider,
        PackageRepository $packageRepository,
        AddressRepository $addressRepository,
        PackageSelectedPointRepository  $packageSelectedPointRepository
    )
    {
        $this->factory = $factory;
        $this->packageProvider = $packageProvider;
        $this->packageRepository = $packageRepository;
        $this->addressRepository = $addressRepository;
        $this->packageSelectedPointRepository = $packageSelectedPointRepository;
    }
    public function generateLabel( GenerateLabelSettings $settings)
    {
        try {
            $this->_checkAddress();
            $this->_createClient();
            $response = $this->_sendRequest($this->_createRequest($settings));
            $labelName = $this->_saveLabel($response['label'], $settings);
            $this->_saveData($response['data'], $labelName, $settings);
        }catch (\Exception $e) {
            throw  $e;
        }
        return true;

    }

    private function _createRequest(GenerateLabelSettings $settings): GenerateLabelBusinessPackListTwo
    {
        $arrayOfBusinessPack = $this->packageProvider->getData($settings);
        if (count($arrayOfBusinessPack->getBusinessPack()) < 1){
            throw new NoPackageException('No packages to generate label');
        }

        $this->client = $this->factory->factory();

        $generateLabel = new GenerateLabelBusinessPackListTwo(
            $this->client->getCredentials()->getPartnerId(),
            $this->client->getCredentials()->getPartnerKey(),
            '',
            $settings->getSettings()['format'],
            $arrayOfBusinessPack->getBusinessPack()

        );
        return $generateLabel;
    }

    private function _createClient()
    {
        $this->client = $this->factory->factory();
    }

    private function _sendRequest(GenerateLabelBusinessPackListTwo $request)
    {
        $response = $this->client->generateLabelBusinessPackListTwo($request);
        $response->getLabelData();
        $data = $response->getGenerateLabelBusinessPackListTwoResult()->geXml(false);
        return ['label' => $response->getLabelData(), 'data' => $data];
    }

    private function _saveData($data, $labelName, GenerateLabelSettings $settings)
    {
        $oneCorrect = false;
        $settingsData = $settings->getSettings();
        foreach ($data->NewDataSet[0]->GenerateLabelBusinessPackListTwo as $index => $responseObject) {
            $this->result[$responseObject->Err][] = ['id' => $settingsData['package_order'][$index], 'description' => $responseObject->ErrDes];
            if (!in_array($responseObject->Err, self::SUCCESS_CODE)){
                continue;
            }
            $oneCorrect = true;

            $settingsData['package_order'][$index];
            $package = $this->packageRepository->find($settingsData['package_order'][$index]);
            $package->setPackCode($responseObject->PackCode_RUCH);
            $package->setPartnerId($this->client->getCredentials()->getPartnerId());
            $package->setLabelName($labelName);
            $package->setLabelDateGenerated(new \DateTime());
            if (isset($settingsData['provider']['form_data']['BoxSize'])){
                $package->setSize($settingsData['provider']['form_data']['BoxSize']);
            }
            if (isset($settingsData['provider']['form_data']['DestinationCode'])){
                $pickup = $this->packageSelectedPointRepository->findOneBy(['destinationCode' => $settingsData['provider']['form_data']['DestinationCode']]);
                if (!empty($pickup)){
                    $package->setPickup($pickup);;
                }
            }
            $package->setSenderOrders($settingsData['sender_orders'][$package->getId()]);
            $protocol = $package->getProtocol();
            if (!empty($protocol)){
                $protocol->removePackages();

            }
            $order = new \Order($package->getIdOrder());
            $orderCarrier = new \OrderCarrier($order->getIdOrderCarrier());
            $orderCarrier->tracking_number = $package->getPackCode();
            $orderCarrier->save();
            $this->packageRepository->add($package, false);
        }
        if ($oneCorrect){
            $this->packageRepository->flush();
        }
    }

    /**
     * Ensure label directory exists, create if not
     *
     * @return bool
     */
    private function _ensureLabelDirectoryExists()
    {
        if (!is_dir(ConfigurationModule::LABEL_PATH)) {
            return mkdir(ConfigurationModule::LABEL_PATH, 0755, true);
        }

        return true;
    }

    /**
     * Save label to file
     *
     * @param string $label
     * @param GenerateLabelSettings $settings
     * @return string
     */
    private function _saveLabel($label, GenerateLabelSettings $settings)
    {
        $this->_ensureLabelDirectoryExists();

        $name = md5(uniqid(rand(), true));
        $format = $settings->getSettings()['format'];
        $ext = $format == 'EPL' ? 'epl' : 'pdf';
        $fullName = $name.'.'.$ext;
        $filename = ConfigurationModule::LABEL_PATH.$fullName;
        file_put_contents($filename, $label);

        return $fullName;
    }

    private function _checkAddress()
    {
        $entity = $this->addressRepository->find(1);
        if (!$entity){
            throw new NoAddresSenderException('No addres sender');
        }
    }

    public function getResult(): array
    {
        return $this->result;
    }

}


