<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Form\Type;

use Orlen\OrlenPaczka\Configuration\Enum\ConfigFormEnum;
use Orlen\OrlenPaczka\Configuration\Enum\FormatEnum;
use Orlen\OrlenPaczka\Configuration\Enum\SenderOrdersTypeEnum;
use Orlen\OrlenPaczka\Configuration\Enum\ShipmentSizeEnum;
use Orlen\OrlenPaczka\Form\Provider\ShipmentCarrierProvider;
use Orlen\OrlenPaczka\Translator\SenderOrdersTypeTranslator;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use PrestaShopBundle\Form\Admin\Type\SwitchType;
use Orlen\OrlenPaczka\Entity\ShipmentType as ShipmentTypeEntity;
use Orlen\OrlenPaczka\Entity\Shipment;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Routing\RouterInterface;
use Orlen\OrlenPaczka\Translator\Adapter\Translator;

class ShipmentType extends AbstractType
{
    private Translator $translator;
    private RouterInterface $router;
    private EntityManagerInterface $entityManager;
    private ShipmentCarrierProvider $carrierProvider;
    private SenderOrdersTypeTranslator $senderOrdersTypeTranslator;


    public function __construct(
        Translator $translator,
        RouterInterface $router,
        EntityManagerInterface $entityManager,
        ShipmentCarrierProvider $carrierProvider,
        SenderOrdersTypeTranslator $senderOrdersTypeTranslator
    )
    {
        $this->translator = $translator;
        $this->router = $router;
        $this->entityManager = $entityManager;
        $this->carrierProvider = $carrierProvider;
        $this->senderOrdersTypeTranslator = $senderOrdersTypeTranslator;
    }

    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $shipment = $builder->getData();
        $currentShipmentTypeId = null;

        if ($shipment instanceof Shipment && $shipment->getShipmentType()) {
            $currentShipmentTypeId = $shipment->getId();
        }

        if ($shipment instanceof Shipment && $shipment->getCarrier()) {
            $currentCarrier = $shipment->getCarrier();
        }

        $builder
            ->add('shipmentType', EntityType::class, [
                'class' => ShipmentTypeEntity::class,
                'label' => $this->translator->trans('Shipment Type', [], 'Modules.Orlenpaczka.Form'),
                'choice_label' => function(ShipmentTypeEntity $shipmentType) {
                    return $shipmentType->getName(); // Dopasuj do twojej encji
                },
//                'query_builder' => function (\Orlen\OrlenPaczka\Repository\ShipmentTypeRepository $repository) use ($currentShipmentTypeId) {
//                    $qb = $repository->createQueryBuilder('st');
//
//                    $usedTypesQb = $this->entityManager->createQueryBuilder()
//                        ->select('IDENTITY(s.shipmentType)')
//                        ->from(Shipment::class, 's')
//                        ->where('s.shipmentType IS NOT NULL');
//
//                    if ($currentShipmentTypeId) {
//                        $usedTypesQb->andWhere('s.id != :currentId');
//                        $qb->where(
//                            'st.id NOT IN (' . $usedTypesQb->getDQL() . ')', // '$usedTypesQb->getDQL()
////
//                        )
//                        ->setParameter('currentId', $currentShipmentTypeId);
//                    } else {
//                        $qb->where($qb->expr()->notIn('st.id', $usedTypesQb->getDQL()));
//                    }
//
//                    return $qb->orderBy('st.name', 'ASC');
//                },
                'placeholder' => 'Wybierz typ przesyłki',
                'required' => true,
            ])
            ->add('carrier', ChoiceType::class, [
                'label' => $this->translator->trans('Carrier', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'choices' => $this->carrierProvider->getCarriers($shipment->getId()),

            ])
            ->add('carrierActive', SwitchType::class, [
                'label' => $this->translator->trans('Carrier Active', [], 'Modules.Orlenpaczka.Form'),
                'required' => false,
            ])
            ->add('mapActive', SwitchType::class, [
                'label' => $this->translator->trans('Map Active', [], 'Modules.Orlenpaczka.Form'),
                'required' => false,

            ])
            ->add('size', ChoiceType::class, [
                'label' => $this->translator->trans('Size', [], 'Modules.Orlenpaczka.Form'),
                'choices' => ShipmentSizeEnum::toArray(),
                'required' => true,
            ])
            ->add('labelFormat', ChoiceType::class, [
                'label' => $this->translator->trans('Label format', [], 'Modules.Orlenpaczka.Form'),
                'choices' => FormatEnum::toArray(),
                'required' => true,
            ])


            ->add('senderOrdersType', ChoiceType::class, [
                'label' => $this->translator->trans('Sender Orders Type', [], 'Modules.Orlenpaczka.Form'),
                'choices' => $this->senderOrdersTypeTranslator->translate(),
                'empty_data' => SenderOrdersTypeEnum::ORDER_REFERENCE,
                'required' => true,
            ])
            ->add('senderOrders', TextType::class, [
                'label' => $this->translator->trans('Sender Orders', [], 'Modules.Orlenpaczka.Form'),
                'required' => false,
                'constraints' => [
                    new \Symfony\Component\Validator\Constraints\Callback([
                        'callback' => function($object, $context) {
                            $form = $context->getRoot();
                            $senderOrdersType = $form->get('senderOrdersType')->getData();

                            if ($senderOrdersType === \Orlen\OrlenPaczka\Configuration\Enum\SenderOrdersTypeEnum::OWN_SENDER_ORDERS && empty($object)) {
                                $context->buildViolation($this->translator->trans('This field is required when using own sender orders', [], 'Modules.Orlenpaczka.Validation'))
                                    ->addViolation();
                            }
                        }
                    ])
                ],
            ]);

        // Dodaje zdarzenie, które automatycznie ustawi insuranceAmount na null, gdy insuranceEnable jest false
//        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event) {
//            $data = $event->getData();
//            if (isset($data['insuranceEnable']) && ($data['insuranceEnable'] === false || $data['insuranceEnable'] === 0 || $data['insuranceEnable'] === '0')) {
//                $data['insuranceAmount'] = null;
//                $event->setData($data);
//            }
//        });


    }

    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults([
            'data_class' => Shipment::class,
        ]);
    }
}
