<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Form\Type;

use Orlen\OrlenPaczka\Configuration\Enum\ConfigFormEnum;
use Orlen\OrlenPaczka\Entity\Address;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Routing\RouterInterface;
use Orlen\OrlenPaczka\Translator\Adapter\Translator;
use Symfony\Component\Validator\Constraints as Assert;

class AddressType extends AbstractType
{
    private Translator $translator;
    private RouterInterface $router;

    public function __construct(Translator $translator, RouterInterface $router)
    {
        $this->translator = $translator;
        $this->router = $router;
    }
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('senderName', TextType::class, [
                'label' => $this->translator->trans('Sender name', [], 'Modules.Orlenpaczka.Form'),
                'constraints' => [
                    new Assert\Length(['max' => '70']),
                ],
                'required' => true,
            ])
            ->add('street', TextType::class, [
                'label' => $this->translator->trans('Street', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '30']),
                ],
            ])
            ->add('buildingNumber', TextType::class, [
                'label' => $this->translator->trans('Building number and apartment', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '10']),
                ],
            ])
            ->add('city', TextType::class, [
                'label' => $this->translator->trans('City', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '30']),
                ],
            ])
            ->add('postalCode', TextType::class, [
                'label' => $this->translator->trans('Postal code', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '6']),
                    new Assert\Regex([
                        'pattern' => '/^[0-9]{2}-[0-9]{3}$/',
                        'message' => $this->translator->trans('Postal code must be in format XX-XXX (e.g. 11-222)', [], 'Modules.Orlenpaczka.Form'),
                    ]),
                ],
            ])
            ->add('country', TextType::class, [
                'label' => $this->translator->trans('Country', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'data' => 'Polska',
                'attr' => [
                    'readonly' => true,
                ],
            ])
            ->add('firstName', TextType::class, [
                'label' => $this->translator->trans('First name', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '30']),
                ],
            ])
            ->add('lastName', TextType::class, [
                'label' => $this->translator->trans('Last name', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '30']),
                ],
            ])
            ->add('phone', TextType::class, [
                'label' => $this->translator->trans('Phone', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '12']),
                    new Assert\Regex([
                        'pattern' => '/^(\+[0-9]{2}[0-9]{9}|[0-9]{9})$/',
                        'message' => $this->translator->trans('Phone number must have exactly 9 digits or 12 digits with country prefix', [], 'Modules.Orlenpaczka.Form'),
                    ]),
                ],
            ])
            ->add('email', TextType::class, [
                'label' => $this->translator->trans('Email', [], 'Modules.Orlenpaczka.Form'),
                'required' => true,
                'constraints' => [
                    new Assert\Length(['max' => '60']),
                    new Assert\Email([
                        'message' => $this->translator->trans('Please enter a valid email address', [], 'Modules.Orlenpaczka.Form'),
                    ]),
                ],
            ]);
    }

    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults([
            'data_class' => Address::class,
        ]);
    }
}
