<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Controller\Admin;

use Orlen\OrlenPaczka\Api\OrlenPaczka\OrlenPaczkaClientFactory;
use Orlen\OrlenPaczka\CallPickup\Provider\PickupDateProvider;
use Orlen\OrlenPaczka\CallPickup\Provider\PickupHoursProvider;
use Orlen\OrlenPaczka\Configuration\ConfigurationModule;
use Orlen\OrlenPaczka\Configuration\Enum\PrintAdressEnum;
use Orlen\OrlenPaczka\Configuration\Enum\PrintTypeEnum;
use Orlen\OrlenPaczka\Exception\IncorrectCredentialsException;
use Orlen\OrlenPaczka\Form\Handler\CallPickupHandler;
use Orlen\OrlenPaczka\Form\Handler\Exception\PackageInDelivery;
use Orlen\OrlenPaczka\Form\Provider\CallPickupProvider;
use Orlen\OrlenPaczka\Form\Provider\FormatLabelProvider;
use Orlen\OrlenPaczka\Form\Type\CallPickupType;
use Orlen\OrlenPaczka\Form\Type\GenerateLabelType;
use Orlen\OrlenPaczka\Label\Exception\NoPackageException;
use Orlen\OrlenPaczka\Label\Generator\LabelGenerator;
use Orlen\OrlenPaczka\Label\Settings\GenerateLabelSettings;
use Orlen\OrlenPaczka\Label\Translator\TranslateResult;
use Orlen\OrlenPaczka\Protocol\Exception\NoPackagesToGenerate;
use Orlen\OrlenPaczka\Protocol\Exception\PackageNoLabel;
use Orlen\OrlenPaczka\Protocol\Exception\ProtocolExist;
use Orlen\OrlenPaczka\Protocol\Generator\ProtocolGenerator;
use Orlen\OrlenPaczka\Protocol\Settings\GenerateProtocolSettings;
use Orlen\OrlenPaczka\Repository\AddressRepository;
use Orlen\OrlenPaczka\Repository\PackageRepository;
use Orlen\OrlenPaczka\Repository\ShipmentRepository;
use Orlen\OrlenPaczka\Search\Filters\PackageFilters;
use Orlen\OrlenPaczka\Tabs\Tab\ShippingHistoryTab;
use Orlen\OrlenPaczka\Tabs\Tabs;
use PrestaShop\PrestaShop\Core\Grid\Presenter\GridPresenterInterface;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Orlen\OrlenPaczka\Translator\Adapter\Translator;

/**
 * @Route(path="/config", name="orlenpaczka.config.shipping")
 */
class ShippingController extends FrameworkBundleAdminController
{
    private Translator $translator;

    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
    }
    /**
     * @Route("/shipping", name=".index", methods={"GET"})
     */

    public function ShippingGridAction(
        \Module $module,
        Tabs $tabs,
        GridPresenterInterface $gridPresenter,
        PackageFilters $filters,
        ShippingHistoryTab $shippingHistoryTab,
        CallPickupProvider $callPickupProvider,
        FormatLabelProvider $formatLabelProvider,
        $gridFactoryPackage
    )
    {
        $shippingHistoryTab->setActive();
        $grid = $gridPresenter->present($gridFactoryPackage->getGrid($filters));

        $formLabel =$this->createForm(GenerateLabelType::class, $formatLabelProvider->getFormatLabelArray(), [
            'url_action' => 'orlenpaczka.config.shipping.generate_label']);
        $formPickup = $this->createForm(CallPickupType::class, $callPickupProvider->getData());


        return $this->render('@Modules/orlenpaczka/views/templates/admin/config/tabs/shipping_history.html.twig', [
            'layoutTitle' => $module->displayName,
            'help_link' => false,
            'tabs' => $tabs,
            'formPickup' => $formPickup->createView(),
            'formLabel' => $formLabel->createView(),
            'grid' => $grid,
            'title' => $this->translator->trans('Shipment', [], 'Modules.Orlenpaczka.Tab'),
        ]);
    }

    /**
     * @Route("/shipping/duplicate/{id}", name=".duplicate", methods={"POST"})
     */

    public function duplicatePackageAction($id, PackageRepository $packageRepository)
    {
        $result = $packageRepository->duplicateForOrder($id);
        if ($result) {
            $this->addFlash('success', $this->translator->trans('New Package created', [], 'Modules.Orlenpaczka.Tab'));
        }else{
            $this->addFlash('error', $this->translator->trans('Package not created', [], 'Modules.Orlenpaczka.Tab'));
        }
        return $this->redirectToRoute('orlenpaczka.config.shipping.index');
    }


    /**
     * @Route("/shipping/generate-label", name=".generate_label", methods={"POST"})
     */

    public function generateLabelAction(
        LabelGenerator $generatorLabelFromPackage,
        Request $request,
        GenerateLabelSettings $settings,
        TranslateResult $translateResult
    )
    {
        $form = $this->createForm(GenerateLabelType::class, [], [
            'url_action' => 'orlenpaczka.config.shipping.generate_label']);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            try{
                $data = $form->getData();
                $settings->setSettings([
                    'provider' => ['id_packages' => $data['packageIds']],
                    'format' => $data['format'],
                    'PrintType' => PrintTypeEnum::ANONYMOUS_ADDRESS,
                    'PrintAdress' => PrintAdressEnum::SENDER_ADDRESS,
                ]);
                $generatorLabelFromPackage->generateLabel($settings);
                $translateResult->translateResult($generatorLabelFromPackage->getResult());
                if ($translateResult->hasErrors() == 0 && $translateResult->hasWarnings() == 0 ) {
                    $this->addFlash('success', $this->translator->trans('Shipment updated', [], 'Modules.Orlenpaczka.Tab'));
                }else{
                    foreach ($translateResult->getSuccess() as $success) {
                        $this->addFlash('success', $success);
                    }
                    foreach ($translateResult->getWarnings() as $warning) {
                        $this->addFlash('warning', $warning);
                    }
                    foreach ($translateResult->getErrors() as $error) {
                        $this->addFlash('error', $error);
                    }
                }
                return $this->redirectToRoute('orlenpaczka.config.shipping.index');
            }
            catch (NoPackageException $e){
                $this->addFlash('error', $this->translator->trans('Error generate label', [], 'Modules.Orlenpaczka.Notification'));
                $this->addFlash('error', $this->translator->trans('No packages to generate label (packages are in delivery or have already been generated label)', [], 'Modules.Orlenpaczka.Notification'));
                return $this->redirectToRoute('orlenpaczka.config.shipping.index');
            }catch (\Exception $e){
                $this->addFlash('error', $this->translator->trans('Error generate label', [], 'Modules.Orlenpaczka.Notification'));
                $this->addFlash('error', $e->getMessage());
                return $this->redirectToRoute('orlenpaczka.config.shipping.index');
            }

        }
    }



    /**
     * @Route("/shipping/download-label/{id}", name=".download_label", methods={"GET"})
     */

    public function downloadLabelAction($id, PackageRepository $packageRepository)
    {
        $package =$packageRepository->find($id);
        $path = ConfigurationModule::LABEL_PATH.DIRECTORY_SEPARATOR. $package->getLabelName();

        if (file_exists($path) && !empty($package->getLabelName())) {
            $response = new \Symfony\Component\HttpFoundation\BinaryFileResponse($path);
            $response->setContentDisposition(
                \Symfony\Component\HttpFoundation\ResponseHeaderBag::DISPOSITION_ATTACHMENT,
                $package->getLabelName()
            );
            return $response;
        }
        $this->addFlash('error', $this->translator->trans('Label not found', [], 'Modules.Orlenpaczka.Tab'));
        return $this->redirectToRoute('orlenpaczka.config.shipping.index');
    }
    /**
     * @Route("/shipping/generate-protocol", name=".generate_protocol", methods={"POST"})
     */

    public function generateProtocolAction(
        ProtocolGenerator $generatorProtocolFromPackage,
        Request $request,
        GenerateProtocolSettings $settings
    )
    {
        $packageIDs = $request->get('orlenpaczka_package_package_bulk');
        $settings->setSettings([
            'provider' => ['id_packages' => $packageIDs],
        ]);
        try{
            $generatorProtocolFromPackage->generateProtocol($settings);
            if (count($generatorProtocolFromPackage->getSuccess()) > 0){
                $this->addFlash('success', $this->translator->trans('Protocol updated', [], 'Modules.Orlenpaczka.Tab'));
            }

            foreach ($generatorProtocolFromPackage->getSuccess() as $row) {
                $this->addFlash('success', $row);
            }

            foreach ($generatorProtocolFromPackage->getErrors() as $row) {
                $this->addFlash('error', $row);
            }
            return $this->redirectToRoute('orlenpaczka.config.shipping.index');
        }
        catch (ProtocolExist $e){
            $this->addFlash('error', $this->translator->trans('Protocol Exist', [], 'Modules.Orlenpaczka.Tab'));
            return $this->redirectToRoute('orlenpaczka.config.shipping.index');
        }
        catch (PackageNoLabel $e){
            $this->addFlash('error', $this->translator->trans('Package no label', [], 'Modules.Orlenpaczka.Tab'));
            return $this->redirectToRoute('orlenpaczka.config.shipping.index');
        }
        catch (NoPackagesToGenerate $e){
            $this->addFlash('error', $this->translator->trans('No packages to generate', [], 'Modules.Orlenpaczka.Tab'));
            return $this->redirectToRoute('orlenpaczka.config.shipping.index');
        }
        catch (\Exception $e){
            $this->addFlash('error', $e->getMessage());
            return $this->redirectToRoute('orlenpaczka.config.shipping.index');
        }

    }

    /**
     * @Route("/shipping/call-pickup/", name=".call_pickup",  methods={"POST"})
     */
    public function callPickupAction(
        Request $request,
        CallPickupHandler $callPickupHandler
    )
    {
        $form = $this->createForm(CallPickupType::class);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            try {
                $callPickupHandler->handle($form->getData());
                $this->addFlash('success', $this->translator->trans('Pickup save', [], 'Modules.Orlenpaczka.Tab'));
                return new JsonResponse(['success' => true, 'redirect' => $this->generateUrl('orlenpaczka.config.shipping.index')]);
            }
            catch (PackageInDelivery $e) {
                return new JsonResponse(['success' => false, 'errors' => [$this->translator->trans('Package in delivery', [], 'Modules.Orlenpaczka.Tab')]]);
            }
            catch (PackageNoLabel $e) {
                return new JsonResponse(['success' => false, 'errors' => [$this->translator->trans('Package no label', [], 'Modules.Orlenpaczka.Tab')]]);
            }catch (\Exception $e) {
                return new JsonResponse(['success' => false, 'errors' => [$e->getMessage()]]);
            }
        }else{
            $errorMessages = [];
            foreach ($form->getErrors(true, true) as $error) {
                $errorMessages[] = $error->getMessage();
            }
            return new JsonResponse(['success' => false, 'errors' => $errorMessages]);
        }

    }

    /**
     * @Route("/shipping/get-date", name=".get_date", methods={"POST"})
     */
    public function getDateAction(
        Request $request,
        PickupDateProvider $dateProvider,
        OrlenPaczkaClientFactory $clientFactory
    ): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $postCode = $data['postCode'] ?? null;
        try{
            $availableDates = $dateProvider->getDates($postCode);
        }catch (IncorrectCredentialsException $e){
            return $this->json([
                'success' => false,
                'message' =>  $e->getMessage(),
            ]);
        }catch (\Exception $e){
            return $this->json([
                'success' => false,
                'message' =>  $this->translator->trans('Incorrect postal code.', [], 'Modules.Orlenpaczka.Form'),
            ]);
        }

        return $this->json([
            'success' => true,
            'dates' => $availableDates,
        ]);

    }

    /**
     * @Route("/shipping/get-hour", name=".get_hour", methods={"POST"})
     */
    public function getHourAction(
        Request $request,
        PickupHoursProvider $hoursProvider
    ): JsonResponse
    {
        $data = json_decode($request->getContent(), true);
        $date = $data['date'] ?? null;
        $postCode = $data['postCode']; // Domyślny kod pocztowy, jeśli nie podano

        if (!$date) {
            return $this->json([
                'success' => false,
                'message' => $this->translator->trans('Date is required', [], 'Modules.Orlenpaczka.Admin'),
            ]);
        }

        try {
            // Pobierz godziny za pomocą providera
            $availableHours = $hoursProvider->getHours($postCode, $date);

            return $this->json([
                'success' => true,
                'hours' => [
                    'from' => $availableHours
                ]
            ]);
        } catch (\Exception $e) {
            return $this->json([
                'success' => false,
                'message' => $this->translator->trans('Failed to fetch available hours: %message%', ['%message%' => $e->getMessage()], 'Modules.Orlenpaczka.Admin'),
            ]);
        }
    }



}
