<?php

declare(strict_types=1);

// phpcs:disable Generic.Files.LineLength.TooLong

namespace Macopedia\HyvaCheckoutOrlen\Magewire;

use Hyva\Checkout\Model\Magewire\Component\Evaluation\EvaluationResult;
use Hyva\Checkout\Model\Magewire\Component\EvaluationInterface;
use Hyva\Checkout\Model\Magewire\Component\EvaluationResultFactory;
use Macopedia\OrlenShipping\Model\Carrier\OrlenShipping;
use Magento\Checkout\Model\Session;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\CartRepositoryInterface;
use Magewirephp\Magewire\Component;
use Macopedia\OrlenShipping\Helper\Data as OrlenShippingHelper;
use Magento\Quote\Model\ResourceModel\Quote as QuoteResource;

class OrlenPaczka extends Component implements EvaluationInterface
{
    /**
     * Loader messages
     *
     * @var array
     */
    protected $loader = [
        'unselectShippingMethod' => 'Saving shipping method',
        'select' => 'Saving shipping method',
    ];

    /**
     * @var string|null
     */
    public ?string $destinationCode = null;

    /**
     * @var string|null
     */
    public ?string $pointDetails = null;

    /**
     * @param Session $session
     * @param CartRepositoryInterface $quoteRepository
     * @param OrlenShippingHelper $orlenShippingHelper
     * @param QuoteResource $quoteResource
     */
    public function __construct(
        protected readonly Session $session,
        protected readonly CartRepositoryInterface $quoteRepository,
        protected readonly OrlenShippingHelper $orlenShippingHelper,
        protected readonly QuoteResource $quoteResource,
    ) {
    }

    /**
     * Boot
     *
     * @return void
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function mount(): void
    {
        $this->destinationCode = $this->session->getQuote()->getData('orlenpaczka_destination_code');
        $this->pointDetails = $this->session->getQuote()->getData('orlenpaczka_point_details');

        $shippingMethod = (string)$this->session->getQuote()?->getShippingAddress()?->getShippingMethod();
        if ($this->destinationCode === null && str_contains($shippingMethod, OrlenShipping::CODE)) {
            $this->unselectShippingMethod();
        }
    }

    /**
     * Select
     *
     * @param string $destinationCode
     * @param string $pointDetails
     * @return void
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function select(string $destinationCode, string $pointDetails): void
    {
        $this->destinationCode = $destinationCode;
        $this->pointDetails = $pointDetails;
        $quote = $this->session->getQuote();

        $quote->setData('orlenpaczka_point_details', $pointDetails);
        $quote->setData('orlenpaczka_destination_code', $destinationCode);
        $quote->getShippingAddress()->setData('shipping_method', 'orlenpaczka_orlenpaczka');
        $this->quoteRepository->save($quote);

        $this->emitToRefresh('price-summary.total-segments');
        $this->emitToRefresh('checkout.shipping.methods');
    }

    /**
     * Evaluate completion
     *
     * @param EvaluationResultFactory $resultFactory
     * @return EvaluationResult
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function evaluateCompletion(EvaluationResultFactory $resultFactory): EvaluationResult
    {
        $shippingMethod = (string)$this->session->getQuote()?->getShippingAddress()?->getShippingMethod();
        if (!str_contains($shippingMethod, OrlenShipping::CODE)) {
            return $resultFactory->createSuccess();
        }

        if ($this->destinationCode === null || $this->pointDetails === null) {
            $errorMessageEvent = $resultFactory->createErrorMessageEvent((string)__('Please select point'))
                ->withCustomEvent('shipping:method:error');
            return $resultFactory->createValidation('validateOrlenPoint')
                ->withFailureResult($errorMessageEvent);
        }

        $quote = $this->session->getQuote();
        $connection = $this->quoteResource->getConnection();
        $connection->update(
            $this->quoteResource->getMainTable(),
            [
                'orlenpaczka_point_details' => $this->pointDetails,
                'orlenpaczka_destination_code' => $this->destinationCode,
            ],
            ['entity_id = ? ' => $quote->getId()]
        );

        return $resultFactory->createSuccess();
    }

    /**
     * Get map widget token
     *
     * @return string
     */
    public function getGeoToken(): string
    {
        return $this->orlenShippingHelper->getMapWidgetToken() ?? '';
    }

    /**
     * Get geo script URL
     *
     * @return string
     */
    public function getGeoScriptUrl(): string
    {
        return 'https://mapa.orlenpaczka.pl/widget.js?token=' . $this->getGeoToken() . '&v=1.0.0';
    }

    /**
     * Unselect shipping method if no locker selected
     *
     * @return void
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function unselectShippingMethod(): void
    {
        $this->destinationCode = null;
        $this->pointDetails = null;

        $quote = $this->session->getQuote();
        $quote->getShippingAddress()->setData('shipping_method', null);
        $this->quoteRepository->save($quote);

        $this->emitToRefresh('checkout.shipping.methods');
    }
}
