using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.AspNetCore.Authorization;
using Microsoft.IdentityModel.Tokens;
using PackEventNotifierOAuth2Example;
using PackEventNotifierOAuth2Example.Models;
using PackEventNotifierOAuth2Example.Services;
using System.Text;
using System.Text.Json;

#region Not required, data used for Frontend
FrontendData frontendData = new();
#endregion

var builder = WebApplication.CreateBuilder(args);

var jwtOptions = builder.Configuration
    .GetSection("JwtOptions")
    .Get<JwtOptions>() ?? throw new Exception("Missing JWT configuration");

builder.Services.AddSingleton(jwtOptions);

builder.Services.AddAuthentication(JwtBearerDefaults.AuthenticationScheme)
    .AddJwtBearer(options =>
    {
        options.RequireHttpsMetadata = false;
        options.SaveToken = true;
        options.TokenValidationParameters = new TokenValidationParameters
        {
            ValidateIssuer = true,
            ValidateAudience = true,
            ValidateLifetime = true,
            ValidateIssuerSigningKey = true,
            ValidIssuer = jwtOptions.Issuer,
            ValidAudience = jwtOptions.Audience,
            IssuerSigningKey = jwtOptions.SymmetricSecurityKey
        };
    });
builder.Services.AddAuthorization();

var app = builder.Build();

app.UseAuthentication();
app.UseRouting();
app.UseAuthorization();

app.MapPost("/token", (HttpContext context, JwtOptions jwtOptions)
    => OAuth2Authorization.Connect(context, jwtOptions));

app.MapPost("/notification", [Authorize] async (HttpRequest request) =>
{
    var body = await new StreamReader(request.Body).ReadToEndAsync();
    var deserialized = JsonSerializer.Deserialize<PackEventNotification>(body);

    // PL: obsuga notyfikacji np. zapis do bazy
    // EN: process notification ex. save to database

    var toLog = (deserialized?.PackCode == 0) ? body : (deserialized?.ToString() ?? "");
    var msg = $"{DateTime.Now:u} {frontendData.Description} {toLog}";
    frontendData.AddPackNotification(msg);
    return frontendData.AcceptPackEventNotifications ? Results.Ok() : Results.NotFound();
});

#region Not required, data used for Frontend
var staticOAuth2Token = OAuth2Authorization.CreateToken("", "userName", "password", false, jwtOptions);

app.MapGet("/", () =>
{
    var manual = File.ReadAllText("Manual.html");
    var authToken = $"{staticOAuth2Token.TokenType} {staticOAuth2Token.AccessToken}";
    return Results.Content(manual.Replace("#token#", authToken), "text/html");
});

app.MapGet("/received", () =>
{
    return Results.Content(string.Join("<br/>", frontendData.ReceivedNotifications), "text/html");
});

app.MapPost("/accept", async (HttpRequest request) =>
{
    frontendData.AcceptPackEventNotifications = await new StreamReader(request.Body).ReadToEndAsync() == "True";
});

app.MapGet("/postman", (string link) =>
{
    var fileName = "OrlenPaczkaPackEventNotifierExampleOAuth2.postman_collection.json";
    var content = File.ReadAllText(fileName).Replace("https://localhost:7047/", link);
    return Results.File(Encoding.UTF8.GetBytes(content), "text/json", $"{fileName}");

});
#endregion

app.Run();
