﻿using PackEventNotifierOAuth2Example.Models;
using System.IdentityModel.Tokens.Jwt;
using System.Security.Claims;

namespace PackEventNotifierOAuth2Example;

public static partial class OAuth2Authorization
{
    public static async Task<IResult> Connect(
        HttpContext context,
        JwtOptions jwtOptions)
    {
        if (context.Request.ContentType != "application/x-www-form-urlencoded")
            return Results.BadRequest(new { Error = "Invalid Request" });

        var formCollection = await context.Request.ReadFormAsync();

        // pulls information from the form
        if (formCollection.TryGetValue("grant_type", out var grantType) == false)
            return Results.BadRequest(new { Error = "Invalid Request" });

        if (formCollection.TryGetValue("username", out var userName) == false)
            return Results.BadRequest(new { Error = "Invalid Request" });

        if (formCollection.TryGetValue("password", out var password) == false)
            return Results.BadRequest(new { Error = "Invalid Request" });

        return Results.Ok(CreateToken(grantType, userName, password, true, jwtOptions));
    }

    public static OAuth2Token CreateToken(
        string grantType, string userName,
        string password, bool expires, JwtOptions jwtOptions)
    {
        var expireIn_s = expires ? jwtOptions.Expiration : TimeSpan.FromDays(100 * 365); // 100 years = lifetime token

        var claims = new List<Claim>()
        {
            new("sub", userName),
            new("name", userName),
            new("aud", jwtOptions.Audience)
        };

        var token = new JwtSecurityToken(
            issuer: jwtOptions.Issuer,
            audience: jwtOptions.Audience,
            claims: claims,
            expires: DateTime.Now.Add(expireIn_s),
            signingCredentials: jwtOptions.SigningCredentials);

        return new OAuth2Token()
        {
            AccessToken = new JwtSecurityTokenHandler().WriteToken(token),
            ExpiresIn = (int)expireIn_s.TotalSeconds,
            TokenType = "Bearer"
        };
    }
}