/**
 * Form utilities for Orlen Paczka module
 */

/**
 * Show error message for a field
 * @param {HTMLElement} field - The form field element
 * @param {string} message - Error message to display
 */
export const showErrorMessage = (field, message) => {
  const formGroup = field.closest('.form-group');
  if (!formGroup) return;

  let errorElement = formGroup.querySelector('.invalid-feedback');

  if (!errorElement) {
    errorElement = document.createElement('div');
    errorElement.className = 'invalid-feedback d-block';

    const inputContainer = field.closest('.col-sm') || formGroup;
    inputContainer.appendChild(errorElement);
  }

  errorElement.textContent = message;
  errorElement.style.display = 'block';

  field.classList.add('is-invalid');
};

/**
 * Hide error message for a field
 * @param {HTMLElement} field - The form field element
 */
export const hideErrorMessage = (field) => {
  field.classList.remove('is-invalid');

  const formGroup = field.closest('.form-group');
  if (formGroup) {
    const errorElements = formGroup.querySelectorAll('.invalid-feedback, .text-danger, .help-block, [class*="error"]');
    errorElements.forEach(element => {
      element.style.display = 'none';
      element.innerHTML = '';
    });

    const colElement = formGroup.querySelector('.col-sm');
    if (colElement) {
      const colErrorElements = colElement.querySelectorAll('.invalid-feedback, .text-danger, .help-block, [class*="error"]');
      colErrorElements.forEach(element => {
        element.style.display = 'none';
        element.innerHTML = '';
      });
    }
  }

  const form = field.closest('form');
  if (form) {
    const fieldId = field.id;
    if (fieldId) {
      const relatedErrors = form.querySelectorAll(`[id$="${fieldId}-error"], [data-for="${fieldId}"]`);
      relatedErrors.forEach(element => {
        element.style.display = 'none';
        element.innerHTML = '';
      });
    }
  }
};

/**
 * Hide all errors in a form
 * @param {HTMLFormElement} form - The form element
 */
export const hideAllErrors = (form) => {
  const invalidFields = form.querySelectorAll('.is-invalid');
  invalidFields.forEach(field => {
    field.classList.remove('is-invalid');
  });

  const errorElements = form.querySelectorAll('.invalid-feedback, .text-danger, .help-block, [class*="error"]');
  errorElements.forEach(element => {
    element.style.display = 'none';
    element.innerHTML = '';
  });
};

/**
 * Create a general error message div
 * @param {string} message - Error message to display
 * @returns {HTMLElement} - The error div element
 */
export const createGeneralErrorDiv = (message) => {
  const errorDiv = document.createElement('div');
  errorDiv.className = 'alert alert-danger';
  errorDiv.textContent = message;
  return errorDiv;
};

/**
 * Add asterisks to labels of required fields
 * @param {HTMLFormElement} form - The form to process
 */
export const markRequiredFields = (form) => {
  if (!form) return;
  console.log('Marking required fields in form', form);
  
  // Direct insertion of style for required fields
  const styleId = 'required-fields-style';
  if (!document.getElementById(styleId)) {
    const style = document.createElement('style');
    style.id = styleId;
    style.textContent = `
      .form-group.required label:after,
      .form-group label.required:after {
        content: ' *';
        color: #ff0000;
        font-weight: bold;
      }
    `;
    document.head.appendChild(style);
  }
  
  // Mark all form groups containing required inputs
  const requiredInputs = form.querySelectorAll('[required], select[data-required="true"]');
  requiredInputs.forEach(input => {
    const formGroup = input.closest('.form-group');
    if (formGroup) {
      formGroup.classList.add('required');
    }
  });
  
  // Specifically mark these important fields as required
  const importantFields = [
    'PostCode',
    'pickup_date',
    'pickup_hours'
  ];
  
  importantFields.forEach(fieldName => {
    // Try different selector patterns
    const selectors = [
      `#orlen_pickup_${fieldName}`,
      `[name="orlen_pickup[${fieldName}]"]`,
      `[id$="_${fieldName}"]`,
      `[name$="[${fieldName}]"]`
    ];
    
    for (const selector of selectors) {
      const field = form.querySelector(selector);
      if (field) {
        // Mark the field's form group as required
        const formGroup = field.closest('.form-group');
        if (formGroup) {
          formGroup.classList.add('required');
          console.log(`Marked field ${fieldName} as required`, field);
        }
        // Also add the required attribute to ensure validation works
        field.setAttribute('required', 'required');
        break;
      }
    }
  });
  
  // If no fields were marked using the above methods, fallback to direct label marking
  const formGroups = form.querySelectorAll('.form-group');
  formGroups.forEach(group => {
    const label = group.querySelector('label');
    const input = group.querySelector('input, select, textarea');
    
    // If we have a label but the group wasn't already marked as required
    if (label && !group.classList.contains('required')) {
      // Mark key fields that should always be required
      const labelText = label.textContent.toLowerCase();
      if (labelText.includes('kod pocztowy') || 
          labelText.includes('data') || 
          labelText.includes('godzina')) {
        group.classList.add('required');
        if (input) {
          input.setAttribute('required', 'required');
        }
      }
    }
  });
};
