<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Soap\Client\Caller;

use Orlen\OrlenPaczka\Soap\Client\Type\RequestInterface;
use Orlen\OrlenPaczka\Soap\Client\Type\ResultInterface;
use Psr\Log\LoggerInterface;

final class LoggingCaller implements CallerInterface
{
    private $caller;
    private $logger;

    public function __construct(CallerInterface $caller, LoggerInterface $logger)
    {
        $this->caller = $caller;
        $this->logger = $logger;
    }

    public function call(string $method, RequestInterface $request): ResultInterface
    {
        $this->logger->info('SOAP request', [
            'method' => $method,
            'params' => $this->serializeRequestForLogging($request),
        ]);

        try {
            $response = $this->caller->call($method, $request);

            $this->logger->info('SOAP response', [
                'response' => $this->serializeResponseForLogging($response),
            ]);

            return $response;
        } catch (\Throwable $e) {
            $this->logger->error('SOAP error', [
                'method' => $method,
                'params' => $this->serializeRequestForLogging($request),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            throw $e;
        }
    }

    /**
     * Serializuje obiekt Request do tablicy dla celów logowania
     *
     * @param RequestInterface $request
     * @return array
     */
    private function serializeRequestForLogging(RequestInterface $request): array
    {
        $result = [
            'class' => get_class($request),
        ];

        $reflect = new \ReflectionObject($request);
        $props = $reflect->getMethods(\ReflectionMethod::IS_PUBLIC);

        foreach ($props as $prop) {
            $name = $prop->getName();
            if (strpos($name, 'get') === 0 && $name !== '__get' && $name !== 'getPartnerKey' && $name !== 'getPartnerID') {
                try {
                    $value = $request->{$name}();
                    if (is_object($value)) {
                        $result[$name] = 'Object: ' . get_class($value);
                    } else {
                        $result[$name] = $value;
                    }
                } catch (\Throwable $e) {
                }
            }
        }

        return $result;
    }

    /**
     * Serializuje obiekt Response do tablicy dla celów logowania
     *
     * @param ResultInterface $response
     * @return array
     */
    private function serializeResponseForLogging(ResultInterface $response): array
    {
        $result = [
            'class' => get_class($response),
        ];

        $reflect = new \ReflectionObject($response);
        $methods = $reflect->getMethods(\ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $name = $method->getName();
            if (strpos($name, 'get') === 0 && $name !== '__get' && $name !== 'getPartnerKey' && $name !== 'getPartnerID') {
                try {
                    $value = $response->{$name}();
                    if (is_object($value)) {
                        if (strpos($name, 'Result') !== false) {
                            $result[$name] = $this->extractResultData($value);
                        } else {
                            $result[$name] = 'Object: ' . get_class($value);
                        }
                    } else {
                        $result[$name] = $value;
                    }
                } catch (\Throwable $e) {
                }
            }
        }

        return $result;
    }

    /**
     * Wyciąga dane z obiektu Result
     *
     * @param object $resultObject
     * @return array
     */
    private function extractResultData($resultObject): array
    {
        $data = [
            'class' => get_class($resultObject),
        ];
        $reflect = new \ReflectionObject($resultObject);
        $methods = $reflect->getMethods(\ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $name = $method->getName();
            if (strpos($name, 'get') === 0 && $name !== '__get' && $name !== 'getPartnerKey' && $name !== 'getPartnerID') {
                try {
                    $value = $resultObject->{$name}();
                    if (is_object($value)) {
                        $data[$name] = 'Object: ' . get_class($value);
                    } else {
                        $data[$name] = $value;
                    }
                } catch (\Throwable $e) {
                }
            }
        }

        return $data;
    }
}
