<?php

namespace Orlen\OrlenPaczka\Presenter\Order;

use Orlen\OrlenPaczka\Entity\Package;
use Orlen\OrlenPaczka\Form\Factory\GenerateLabelOrderFormFactory;
use Orlen\OrlenPaczka\Form\Type\GenerateLabelOrderType;
use Orlen\OrlenPaczka\Repository\PackageRepository;
use Orlen\OrlenPaczka\View\OrderView;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Routing\RouterInterface;
use Twig\Environment;

class AdminOrderPresenter
{
    private PackageRepository $packageRepository;
    private Environment $twig;
    private RouterInterface $router;
    private FormFactoryInterface $formFactory;
    private GenerateLabelOrderFormFactory $generateLabelOrderFormFactory;
    private $translator;

    public function __construct(
        PackageRepository $packageRepository,
        Environment $twig,
        RouterInterface $router,
        FormFactoryInterface $formFactory,
        GenerateLabelOrderFormFactory $generateLabelOrderFormFactory,
        $translator
    )
    {
        $this->packageRepository = $packageRepository;
        $this->twig = $twig;
        $this->router = $router;
        $this->formFactory = $formFactory;
        $this->generateLabelOrderFormFactory = $generateLabelOrderFormFactory;
        $this->translator = $translator;
    }
    /**
     * Present order packages
     *
     * @param int $idOrder Order ID
     *
     * @return string Rendered template with package data
     */
    public function present(int $idOrder)
    {
        $packages = $this->packageRepository->findBy(['idOrder' => $idOrder]);
        $order = new \Order($idOrder);


        $products = $order->getProducts();
        // Extract weights and sum them
        $weights = array_column($products, 'weight');

        // Sprawdzamy czy istnieje jakiś produkt z wagą 0 (może być string lub liczba)
        $noSetWeight = false;
        foreach ($weights as $weight) {
            if ((float)$weight == 0) {
                $noSetWeight = true;
                break;
            }
        }

        $totalWeight = (float) array_sum($weights);
//        $totalWeight = 0;
        if($totalWeight > 20) {
            $weight = '<div class="alert alert-danger" role="alert"><strong>'.$this->translator->trans('The order contains products whose total weight exceeds the allowable limit for ORLEN Paczka shipment (20 kg). To split the products, add another shipment to the order.', [], 'Modules.Orlenpaczka.Order').'</strong></div>';
        }elseif($totalWeight == 0 || $noSetWeight) {
            $weight = '<div class="alert alert-warning" role="alert"><strong>'.$this->translator->trans('The order contains products that do not have a defined weight. Make sure that the weight of a single ORLEN Paczka shipment does not exceed 20 kg.', [], 'Modules.Orlenpaczka.Order').'</strong></div>';
        }else{
            $weight =  '';
        }

        if (!empty($packages)) {
            // Przygotowanie obiektów OrderView dla paczek
            $orderViews = $this->_presentPackages($packages, $order);

            // Generowanie formularzy dla wszystkich paczek
            $forms = [];
            foreach ($orderViews as $index => $orderView) {
                if ($orderView->getFormId()) {
                    // Tworzymy formularz dla każdej paczki
                    $formInstance = $this->generateLabelOrderFormFactory->factory($orderView->getFormId());
                    $forms[$orderView->getFormId()] = $formInstance->createView();
                }
            }

            // Renderowanie widoku z tablicą obiektów OrderView i formularzami
            return $this->twig->render('@Modules/orlenpaczka/views/templates/admin/order/orlen_order.html.twig', [
                'orderViews' => $orderViews,
                'forms' => $forms,
                'weight' => $weight
            ]);
        }

        return $this->twig->render('@Modules/orlenpaczka/views/templates/admin/order/orlen_order.html.twig', [
            'orderViews' => [],
            'forms' => [],
            'weight' => $weight
        ]);
    }

    /**
     * Process collection of packages and return collection of OrderView objects
     *
     * @param array $packages Collection of Package entities
     * @param \Order $order Order object
     *
     * @return array Collection of OrderView objects
     */
    private function _presentPackages(array $packages, \Order $order)
    {
        $orderViews = [];

        foreach ($packages as $package) {
            $orderViews[] = $this->_presentPackage($package, $order);
        }

        return $orderViews;
    }

    /**
     * Present single package
     *
     * @param Package $package Package entity
     * @param \Order $order Order object
     *
     * @return OrderView View object with package data
     */
    private function _presentPackage(Package $package, \Order $order)
    {
        $orderView = new OrderView();
        if (!empty($package->getLabelDateGenerated())){
            $orderView->setPackageCode($package->getPackCode());
            $orderView->setPackageDateLabelGenerate($package->getLabelDateGenerated());
            $orderView->sePackageLabelDownloadLink($this->router->generate('orlenpaczka.config.shipping.download_label', ['id' => $package->getId()]));
            $orderView->setPackageGridLink($this->router->generate('orlenpaczka.config.shipping.index', ['orlenpaczka_package[filters][id_order]' => $order->id]));
            $orderView->setPackageTrackingUrl('https://www.orlenpaczka.pl/sledz-paczke/?numer='.$package->getPackCode());
            $orderView->setAllowRegenerateLabel(empty($package->getDelivery()));
        }

        $orderView->setPackageSize($package->getSize());
        $pickup = $package->getPickup();
        $orderView->setDuplicateUrl($this->router->generate('orlenpaczka.order.duplicate', ['id' => $package->getId(), 'idOrder' => $order->id]));
        $orderView->setPickupPoint($pickup->getPoint());
        $orderView->setPickupDestinationCode($pickup->getDestinationCode());
        $orderView->setPickuAddress($pickup->getAddress());
        $orderView->setPickupCity($pickup->getCity());
        $orderView->setPickupPostalCode($pickup->getPostalCode());
        $orderView->setPickupId($pickup->getId());
        ;
        $protocol = $package->getProtocol();
        if ($protocol) {
            $orderView->setProtocolCode($protocol->getCode());
            $orderView->setProtocolLabelDownloadLink($this->router->generate('orlenpaczka.config.protocol.download_label', ['id' => $protocol->getId()]));
            $orderView->setProtocolDateLabelGenerate($protocol->getLabelDateGenerated());
        }

        // Przechowujemy referencję do formularza w OrderView aby mieć do niego dostęp w present()
        $orderView->setFormId($package->getId());

        return $orderView;
    }
}
