<?php

namespace Orlen\OrlenPaczka\Logger\Driver;

use Doctrine\Persistence\ManagerRegistry;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Processor\IntrospectionProcessor;
use Monolog\Processor\PsrLogMessageProcessor;
use Monolog\Processor\UidProcessor;
use Orlen\OrlenPaczka\Configuration\ConfigurationModule;
use Orlen\OrlenPaczka\Configuration\Enum\LoggerStoreEnum;
use Orlen\OrlenPaczka\Entity\Log;
use Orlen\OrlenPaczka\Logger\Handler\DoctrineHandler;
use Orlen\OrlenPaczka\Repository\LogRepository;
use Psr\Log\LoggerInterface;
use Symfony\Component\OptionsResolver\Options;
use Symfony\Component\OptionsResolver\OptionsResolver;

abstract class AbstractDriver
{


    /**
     * Dodaje odpowiednie handlery do loggera na podstawie konfiguracji
     *
     * @param LoggerInterface $logger
     * @param array $options
     * @throws \InvalidArgumentException Jeśli konfiguracja dla bazy danych jest niepełna
     */
    public function setHandler(LoggerInterface $logger, array $options)
    {
        foreach ($options['handlers'] as $handler) {
            switch ($handler) {
                case LoggerStoreEnum::FILE:
                    $logger->pushHandler(new RotatingFileHandler(
                        $options['log_path'] . $this->getName() . '.log',
                        $options['max_file'],
                        $options['log_level']
                    ));
                    break;
                case LoggerStoreEnum::DATABASE:
                    if (null === $this->entityManager) {
                        throw new \InvalidArgumentException('Doctrine ManagerRegistry must be set for database logging');
                    }

                    // Pobieramy repozytorium logów
                    $logRepository = $this->entityManager->getRepository(Log::class);

                    // Sprawdzamy czy repozytorium jest poprawnego typu
                    if (!$logRepository instanceof LogRepository) {
                        throw new \InvalidArgumentException('Invalid repository type');
                    }

                    // Dodajemy handler do loggera
                    $logger->pushHandler(new DoctrineHandler(
                        $logRepository,
                        $options['log_level']
                    ));
                    break;
            }
        }
    }

    public function setProcessor(LoggerInterface $logger, array $options)
    {
//        $logger->pushProcessor(new UidProcessor());
//        $logger->pushProcessor(new IntrospectionProcessor($options['introspection_log_level']));
//        $logger->pushProcessor(new PsrLogMessageProcessor());
    }
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults([
            'handlers' => ConfigurationModule::LOG_STORE,
            'max_file' => ConfigurationModule::LOG_MAX_FILE,
            'log_level' => ConfigurationModule::LOG_LEVEL,
            'introspection_log_level' => ConfigurationModule::LOG_INTROSPECTION_LEVEL,
            'log_path' => ConfigurationModule::LOG_PATH,
        ]);

        $resolver->setRequired(['handlers', 'max_file', 'log_level', 'log_path', 'introspection_log_level']);

        $resolver->setAllowedTypes('handlers', 'array');
        $resolver->setAllowedTypes('max_file', 'int');
        $resolver->setAllowedTypes('log_level', 'int');
        $resolver->setAllowedTypes('introspection_log_level', 'int');
        $resolver->setAllowedTypes('log_path', 'string');

        $resolver->setNormalizer('handlers', function (Options $options, $value) {
            $loggerStoreEnum = new LoggerStoreEnum();
            $allowedValues = $loggerStoreEnum->getCases();

            foreach ($value as $key => $handler) {
                if (!in_array($handler, $allowedValues, true)) {
                    $allowedValuesString = implode(', ', $allowedValues);
                    throw new \InvalidArgumentException(
                        sprintf(
                            'Handlers[%s] must be one of the allowed LoggerStoreEnum values (%s), got: %s',
                            $key,
                            $allowedValuesString,
                            is_object($handler) ? get_class($handler) : gettype($handler) . ' ' . (is_scalar($handler) ? $handler : '')
                        )
                    );
                }
            }

            return $value;
        });
    }
}
