<?php

namespace Orlen\OrlenPaczka\Grid\Query;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Orlen\OrlenPaczka\Configuration\Enum\SenderOrdersTypeEnum;
use Orlen\OrlenPaczka\Translator\SenderOrdersTypeTranslator;
use PrestaShop\PrestaShop\Core\Grid\Query\AbstractDoctrineQueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\DoctrineSearchCriteriaApplicatorInterface;
use PrestaShop\PrestaShop\Core\Grid\Search\SearchCriteriaInterface;

class ShipmentQueryBuilder extends AbstractDoctrineQueryBuilder
{
    private $searchCriteriaApplicator;

    private SenderOrdersTypeTranslator $senderOrdersTypeTranslator;

    public function __construct(
        Connection $connection,
        string $dbPrefix,
        DoctrineSearchCriteriaApplicatorInterface $searchCriteriaApplicator,
        SenderOrdersTypeTranslator $senderOrdersTypeTranslator
    ) {
        parent::__construct($connection, $dbPrefix);

        $this->searchCriteriaApplicator = $searchCriteriaApplicator;
        $this->senderOrdersTypeTranslator = $senderOrdersTypeTranslator;
    }

    public function getSearchQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select(
                'os.*,
                 CASE os.sender_orders_type
                    WHEN 0 THEN os.sender_orders
                    WHEN 1 THEN :one
                    WHEN 2 THEN :two
                    WHEN 3 THEN :three
                    ELSE os.sender_orders
                    END as sender_orders
                ');

        $queryBuilder->setParameter('one', $this->senderOrdersTypeTranslator->getTranslationByType(SenderOrdersTypeEnum::ORDER_REFERENCE));
        $queryBuilder->setParameter('two', $this->senderOrdersTypeTranslator->getTranslationByType(SenderOrdersTypeEnum::ORDER_ID));
        $queryBuilder->setParameter('three', $this->senderOrdersTypeTranslator->getTranslationByType(SenderOrdersTypeEnum::NONE));

        $this->searchCriteriaApplicator
            ->applyPagination($searchCriteria, $queryBuilder)
            ->applySorting($searchCriteria, $queryBuilder);

        return $queryBuilder;
    }

    public function getCountQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        return $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select('COUNT(*)');
    }

    private function getCommonQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this->connection
            ->createQueryBuilder()
            ->from($this->dbPrefix . 'orlenpaczka_shipment', 'os')
        ;

        $this->applyFilters($searchCriteria->getFilters(), $queryBuilder);

        return $queryBuilder;
    }

    private function applyFilters(array $filters, QueryBuilder $qb)
    {
        foreach ($filters as $filterName => $filterValue) {
            switch ($filterName) {
                case 'id_shipment':
                case 'id_carrier_reference':
                case 'insurance_amount':
                    $qb->andWhere('os.`' . $filterName . '` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue);

                    break;
                case 'carrier_active':
                case 'map_active':
                    $qb->andWhere('os.`' . $filterName . '` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue ? 1 : 0);

                    break;
                case 'size':
                case 'sender_orders':
                    $qb->andWhere('os.`' . $filterName . '` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');

                    break;
                default:
                    break;
            }
        }
    }
}
