<?php

namespace Orlen\OrlenPaczka\Grid\Query;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Orlen\OrlenPaczka\Configuration\Enum\SenderOrdersTypeEnum;
use Orlen\OrlenPaczka\Entity\PackageSelectedPoint;
use Orlen\OrlenPaczka\Entity\SelectedPoint;
use Orlen\OrlenPaczka\Entity\Shipment;
use Orlen\OrlenPaczka\Entity\ShipmentType;
use Orlen\OrlenPaczka\Translator\SenderOrdersTypeTranslator;
use PrestaShop\PrestaShop\Core\Grid\Query\AbstractDoctrineQueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\DoctrineSearchCriteriaApplicatorInterface;
use PrestaShop\PrestaShop\Core\Grid\Search\SearchCriteriaInterface;

class PackageQueryBuilder extends AbstractDoctrineQueryBuilder
{
    private $searchCriteriaApplicator;
    private SenderOrdersTypeTranslator $senderOrdersTypeTranslator;

    public function __construct(
        Connection $connection,
        string $dbPrefix,
        DoctrineSearchCriteriaApplicatorInterface $searchCriteriaApplicator,
        SenderOrdersTypeTranslator $senderOrdersTypeTranslator
    ) {
        parent::__construct($connection, $dbPrefix);

        $this->searchCriteriaApplicator = $searchCriteriaApplicator;
        $this->senderOrdersTypeTranslator = $senderOrdersTypeTranslator;
    }

    public function getSearchQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {


        $queryBuilder = $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select(
                '
                op.*,
                 CASE
                    WHEN op.label_name IS NULL THEN
                        CASE s.sender_orders_type
                              WHEN 0 THEN s.sender_orders
                              WHEN 1 THEN o.reference
                              WHEN 2 THEN CAST(o.id_order AS CHAR)
                              WHEN 3 THEN :three
                        END
                    ELSE
                        COALESCE(op.sender_orders, :three)
                 END as sender_orders,
                 o.date_add,
                 c.email,
                 a.phone,
                 p.destination_code,
                 IF(op.label_name IS NULL,0,1) AS generate_label,
                 IF(op.id_protocol IS NULL,0,1) AS generate_protocol,
                 IF(op.id_delivery IS NULL,0,1) AS delivery
                       ');

        $this->searchCriteriaApplicator
            ->applyPagination($searchCriteria, $queryBuilder)
            ->applySorting($searchCriteria, $queryBuilder);

        return $queryBuilder;
    }

    public function getCountQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        return $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select('COUNT(*)');
    }

    private function getCommonQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this->connection
            ->createQueryBuilder()
            ->from($this->dbPrefix . 'orlenpaczka_package', 'op')
            ->innerJoin('op', $this->dbPrefix.'orders', 'o', 'op.id_order = o.id_order')
            ->innerJoin('o', $this->dbPrefix.'customer', 'c', 'o.id_customer = c.id_customer')
            ->innerJoin('o', $this->dbPrefix.'address', 'a', 'o.id_address_delivery = a.id_address')
            ->innerJoin('op', Shipment::TABLE_NAME, 's', 'op.id_shipment = s.id_shipment')
            ->innerJoin('op', PackageSelectedPoint::TABLE_NAME, 'p', 'op.id_pickup = p.id_pickup')
//            ->innerJoin('s', ShipmentType::TABLE_NAME, 'st', 's.id_shipment_type = st.id_shipment_type')
            ->orderBy('o.date_add', 'DESC')
        ;
        $queryBuilder->setParameter('one', $this->senderOrdersTypeTranslator->getTranslationByType(SenderOrdersTypeEnum::ORDER_REFERENCE));
        $queryBuilder->setParameter('two', $this->senderOrdersTypeTranslator->getTranslationByType(SenderOrdersTypeEnum::ORDER_ID));
        $queryBuilder->setParameter('three', $this->senderOrdersTypeTranslator->getTranslationByType(SenderOrdersTypeEnum::NONE));


        $this->applyFilters($searchCriteria->getFilters(), $queryBuilder);

        return $queryBuilder;
    }

    private function applyFilters(array $filters, QueryBuilder $qb)
    {
        foreach ($filters as $filterName => $filterValue) {
            switch ($filterName) {
                case 'id_package':
                case 'id_order':
                    $qb->andWhere('op.`' . $filterName . '` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue);
                    break;
                case 'size':
                    $qb->andWhere('s.`size` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue);
                    break;
                case 'insurance_amount':
                    $qb->andWhere('s.`insurance_amount` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue);
                    break;
                case 'pack_code':
                    $qb->andWhere('op.`pack_code` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');
                    break;
                case 'destination_code':
                    $qb->andWhere('p.`destination_code` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');
                    break;
                case 'sender_orders':
                    // Filtruj według tej samej logiki co w SELECT - sprawdzaj wszystkie możliwe źródła danych
                    $qb->andWhere('
                        (
                            (op.label_name IS NULL AND s.sender_orders_type = 0 AND s.sender_orders LIKE :' . $filterName . ') OR
                            (op.label_name IS NULL AND s.sender_orders_type = 1 AND o.reference LIKE :' . $filterName . ') OR
                            (op.label_name IS NULL AND s.sender_orders_type = 2 AND CAST(o.id_order AS CHAR) LIKE :' . $filterName . ') OR
                            (op.label_name IS NULL AND s.sender_orders_type = 3 AND :three LIKE :' . $filterName . ') OR
                            (op.label_name IS NOT NULL AND op.sender_orders LIKE :' . $filterName . ') OR
                            (op.label_name IS NOT NULL AND op.sender_orders IS NULL AND :three LIKE :' . $filterName . ')
                        )
                    ');
                    $qb->setParameter($filterName, '%' . $filterValue . '%');
                    break;
                case 'phone':
                    $qb->andWhere('a.`phone` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');
                    break;
                case 'email':
                    $qb->andWhere('c.`email` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');
                    break;
                case 'date_add':
                    if (isset($filterValue['from'])){
                        $qb->andWhere('o.`date_add` >= :from');
                        $qb->setParameter('from',  $filterValue['from'] );
                    }
                    if (isset($filterValue['to'])){
                        $qb->andWhere('o.`date_add` <= :to');
                        $to = $filterValue['to'].' 23:59:59';
                        $qb->setParameter('to',  $to);
                    }
                    break;
                case 'label_date_generated':
                    if (isset($filterValue['from'])){
                        $qb->andWhere('op.`label_date_generated` >= :from');
                        $qb->setParameter('from',  $filterValue['from'] );
                    }
                    if (isset($filterValue['to'])){
                        $qb->andWhere('op.`label_date_generated` <= :to');
                        $to = $filterValue['to'].' 23:59:59';
                        $qb->setParameter('to',  $to);
                    }

                    break;
                case 'partner_id':
                    $qb->andWhere('op.`' . $filterName . '` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');
                    break;
                case 'generate_protocol':
                    if ($filterValue == 1) {
                        $qb->andWhere('op.`id_protocol` IS NOT NULL');
                    }else{
                        $qb->andWhere('op.`id_protocol` IS NULL OR op.`label_name` = :' . $filterName);
                        $qb->setParameter($filterName,  '');
                    }

                    break;
                case 'delivery':
                    if ($filterValue == 1) {
                        $qb->andWhere('op.`id_delivery` IS NOT NULL');
                    }else{
                        $qb->andWhere('op.`id_delivery` IS NULL');
                    }

                    break;
                case 'generate_label':
                    if ($filterValue == 1) {
                        $qb->andWhere('op.`label_name` IS NOT NULL');
                    }else{
                        $qb->andWhere('op.`label_name` IS NULL OR op.`label_name` = :' . $filterName);
                        $qb->setParameter($filterName,  '');
                    }

                    break;
                default:
                    break;
            }
        }
    }
}
