<?php

namespace Orlen\OrlenPaczka\Grid\Query;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\AbstractDoctrineQueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\DoctrineSearchCriteriaApplicatorInterface;
use PrestaShop\PrestaShop\Core\Grid\Search\SearchCriteriaInterface;

class DeliveryQueryBuilder extends AbstractDoctrineQueryBuilder
{
    private $searchCriteriaApplicator;

    public function __construct(
        Connection $connection,
        string $dbPrefix,
        DoctrineSearchCriteriaApplicatorInterface $searchCriteriaApplicator
    ) {
        parent::__construct($connection, $dbPrefix);

        $this->searchCriteriaApplicator = $searchCriteriaApplicator;
    }

    public function getSearchQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select(
                'od.*,
                op.id_protocol,
                 GROUP_CONCAT(opa.pack_code) as packages,
                 op.protocol_code as protocol_code,
                 IF(op.label_date_generated IS NULL,"0000-00-00 00:00:00",op.label_date_generated) AS protocol_create
                 ')
            ->groupBy('od.id_delivery');

        $this->searchCriteriaApplicator
            ->applyPagination($searchCriteria, $queryBuilder)
            ->applySorting($searchCriteria, $queryBuilder);

        return $queryBuilder;
    }

    public function getCountQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        return $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select('COUNT(*)');
    }

    private function getCommonQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this->connection
            ->createQueryBuilder()
            ->from($this->dbPrefix . 'orlenpaczka_delivery', 'od')
            ->innerJoin('od',$this->dbPrefix . 'orlenpaczka_package', 'opa', 'od.id_delivery = opa.id_delivery')
            ->leftJoin('od',$this->dbPrefix . 'orlenpaczka_protocol', 'op', 'od.id_delivery = op.id_delivery')
        ;

        $this->applyFilters($searchCriteria->getFilters(), $queryBuilder);

        return $queryBuilder;
    }

    private function applyFilters(array $filters, QueryBuilder $qb)
    {
        foreach ($filters as $filterName => $filterValue) {
            switch ($filterName) {
                case 'id_delivery':
                    $qb->andWhere('od.`' . $filterName . '` = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue);

                    break;
                case 'code':
                    $qb->andWhere('od.`' . $filterName . '` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');

                    break;
                case 'date_pickup':
                case 'date_add':
                    if (isset($filterValue['from'])) {
                        $qb->andWhere('od.`' . $filterName . '` >= :' . $filterName . '_from');
                        $qb->setParameter($filterName . '_from', sprintf('%s 0:0:0', $filterValue['from']));
                    }

                    if (isset($filterValue['to'])) {
                        $qb->andWhere('od.`' . $filterName . '` <= :' . $filterName . '_to');
                        $qb->setParameter($filterName . '_to', sprintf('%s 23:59:59', $filterValue['to']));
                    }

                    break;
                case 'hour_pickup_from':
                case 'hour_pickup_to':
                    if (isset($filterValue['from'])) {
                        $qb->andWhere('od.`' . $filterName . '` >= :' . $filterName . '_from');
                        $qb->setParameter($filterName . '_from', sprintf('%s:00:00', $filterValue['from']));
                    }

                    if (isset($filterValue['to'])) {
                        $qb->andWhere('od.`' . $filterName . '` <= :' . $filterName . '_to');
                        $qb->setParameter($filterName . '_to', sprintf('%s:00:00', $filterValue['to']));
                    }

                    break;
                case  'packages':
                    $qb->andHaving('GROUP_CONCAT(opa.pack_code) LIKE :packages');
                    $qb->setParameter('packages', '%' . $filterValue . '%');
                case 'protocol_create':
                    if (isset($filterValue['from'])) {
                        $qb->andWhere('op.`label_date_generated` >= :' . $filterName . '_from');
                        $qb->setParameter($filterName . '_from', sprintf('%s 0:0:0', $filterValue['from']));
                    }

                    if (isset($filterValue['to'])) {
                        $qb->andWhere('op.`label_date_generated` <= :' . $filterName . '_to');
                        $qb->setParameter($filterName . '_to', sprintf('%s 23:59:59', $filterValue['to']));
                    }

                    break;
                case 'protocol_code':
                    $qb->andWhere('op.`' . $filterName . '` LIKE :' . $filterName);
                    $qb->setParameter($filterName, '%' . $filterValue . '%');

                    break;
                default:
                    break;
            }
        }
    }
}
