<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Form\Type;

use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\CallbackTransformer;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\CollectionType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Routing\RouterInterface;
use Orlen\OrlenPaczka\Translator\Adapter\Translator;
use Symfony\Component\Validator\Constraints as Assert;

class CallPickupType extends AbstractType
{
    private RouterInterface $router;
    private Translator $translator;

    public function __construct(RouterInterface $router, Translator $translator)
    {
        $this->router = $router;
        $this->translator = $translator;
    }
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder->setAction($this->router->generate('orlenpaczka.config.shipping.call_pickup'));
        // Ukryte pole do przechowywania ID paczek oddzielonych przecinkami
        $builder->add('packageIds', CollectionType::class, [
            'allow_add' => true,
            'entry_type' => HiddenType::class,
            'label' => false,
    ])
        ->get('packageIds')
        ->addModelTransformer(new CallbackTransformer(
            static function ($packageIds) {
                return $packageIds;
            },
            static function (array $packageIds) {
                return array_map(static function ($packageIds) {
                    return (int) $packageIds;
                }, $packageIds);
            }
        ));

        $builder->add('PostCode', TextType::class, [
            'required' => true,
            'label' => $this->translator->trans('Post Code', [], 'Modules.Orlenpaczka.Form'),
            'constraints' => [
                new Assert\NotBlank([
                    'message' => $this->translator->trans('Post code field is required', [], 'Modules.Orlenpaczka.Form')
                ]),
                new Assert\Regex([
                    'pattern' => '/^\d{2}-\d{3}$/',
                    'message' => $this->translator->trans('Postal code must be in format XX-XXX (e.g. 11-222)', [], 'Modules.Orlenpaczka.Form')
                ])
            ]
        ]);


        // Select do wyboru daty
        $builder->add('pickup_date', ChoiceType::class, [
            'choices' => [],
            'required' => true,
            'attr' => [
                'class' => 'pickup-date-select',
            ],
            'label' => $this->translator->trans('Date', [], 'Modules.Orlenpaczka.Form'),
        ]);

        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event) {
            $form = $event->getForm();
            $data = $event->getData();

            // Jeśli mamy wartość dla pickup_date w danych formularza
            if (isset($data['pickup_date'])) {
                $pickupDate = $data['pickup_date'];

                // Dynamicznie dodajemy tę wartość jako dozwoloną opcję
                $form->add('pickup_date', ChoiceType::class, [
                    'choices' => [$pickupDate => $pickupDate],
                    'required' => true,
                    'attr' => [
                        'class' => 'pickup-date-select',
                    ],
                    'label' => $this->translator->trans('Date', [], 'Modules.Orlenpaczka.Form'),
                ]);
            }
        });

        // Puste selecty na godziny - będą uzupełniane przez AJAX
        $builder->add('pickup_hours', ChoiceType::class, [
            'choices' => [],
            'placeholder' => 'Wybierz godzinę od',
            'required' => true,

            'attr' => [
                'class' => 'pickup-hour-from-select',
                'disabled' => 'disabled',
            ],
            'label' => $this->translator->trans('Hour', [], 'Modules.Orlenpaczka.Form'),
        ]);

        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event) {
            $form = $event->getForm();
            $data = $event->getData();

            // Jeśli mamy wartość dla pickup_date w danych formularza
            if (isset($data['pickup_hours'])) {
                $pickupDate = $data['pickup_hours'];

                // Dynamicznie dodajemy tę wartość jako dozwoloną opcję
                $form->add('pickup_hours', ChoiceType::class, [
                    'choices' => [$pickupDate => $pickupDate],
                    'required' => true,
                    'attr' => [
                        'class' => 'pickup-hour-from-select',
                    ],
                    'label' => $this->translator->trans('Hour', [], 'Modules.Orlenpaczka.Form'),
                ]);
            }
        });


        $builder->add('City', TextType::class, [
            'required' => true,
            'label' => $this->translator->trans('City', [], 'Modules.Orlenpaczka.Form'),
            'constraints' => [
                new Assert\NotBlank([
                    'message' => $this->translator->trans('City field is required', [], 'Modules.Orlenpaczka.Form')
                ])
            ]
        ]);

        $builder->add('Street', TextType::class, [
            'required' => true,
            'label' => $this->translator->trans('Street', [], 'Modules.Orlenpaczka.Form'),
            'constraints' => [
                new Assert\NotBlank([
                    'message' => $this->translator->trans('Street field is required', [], 'Modules.Orlenpaczka.Form')
                ])
            ]
        ]);

        $builder->add('BulidingNo', TextType::class, [
            'required' => false,
            'label' => $this->translator->trans('Building number', [], 'Modules.Orlenpaczka.Form')
        ]);

        $builder->add('Email', EmailType::class, [
            'required' => true,
            'label' => $this->translator->trans('Email', [], 'Modules.Orlenpaczka.Form'),
            'constraints' => [
                new Assert\NotBlank([
                    'message' => $this->translator->trans('Email field is required', [], 'Modules.Orlenpaczka.Form')
                ]),
                new Assert\Email([
                    'message' => $this->translator->trans('Please enter a valid email address', [], 'Modules.Orlenpaczka.Form')
                ])
            ]
        ]);

        $builder->add('PartnerName', TextType::class, [
            'required' => true,
            'label' => $this->translator->trans('Partner Name', [], 'Modules.Orlenpaczka.Form'),
            'constraints' => [
                new Assert\NotBlank([
                    'message' => $this->translator->trans('Partner Name field is required', [], 'Modules.Orlenpaczka.Form')
                ])
            ]
        ]);

        $builder->add('PersonName', TextType::class, [
            'required' => false,
            'label' => $this->translator->trans('Person Name', [], 'Modules.Orlenpaczka.Form'),
        ]);

        $builder->add('PersonSurname', TextType::class, [
            'required' => false,
            'label' => $this->translator->trans('Person Surname', [], 'Modules.Orlenpaczka.Form'),
        ]);

        $builder->add('Telephone', TextType::class, [
            'required' => false,
            'label' => $this->translator->trans('Telephone', [], 'Modules.Orlenpaczka.Form'),
        ]);

    }

    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults([
            'data_class' => null,
            'csrf_protection' => true,
            'csrf_field_name' => '_token',
            'csrf_token_id' => 'pickup_form',
            'attr' => [
                'class' => 'pickup-form',
                'data-pickup-form' => true,
            ],
        ]);
    }

    public function getBlockPrefix()
    {
        return 'orlen_pickup';
    }
}
