<?php

declare(strict_types=1);

namespace Orlen\OrlenPaczka\Controller\Admin;

use Orlen\OrlenPaczka\Configuration\Enum\PrintAdressEnum;
use Orlen\OrlenPaczka\Configuration\Enum\PrintTypeEnum;
use Orlen\OrlenPaczka\Entity\PackageSelectedPoint;
use Orlen\OrlenPaczka\Form\Factory\GenerateLabelOrderFormFactory;
use Orlen\OrlenPaczka\Form\Type\GenerateLabelOrderType;
use Orlen\OrlenPaczka\Form\Type\GenerateLabelType;
use Orlen\OrlenPaczka\Label\Exception\NoPackageException;
use Orlen\OrlenPaczka\Label\Generator\LabelGenerator;
use Orlen\OrlenPaczka\Label\Settings\GenerateLabelSettings;
use Orlen\OrlenPaczka\Label\Translator\TranslateResult;
use Orlen\OrlenPaczka\Repository\PackageRepository;
use Orlen\OrlenPaczka\Repository\PackageSelectedPointRepository;
use Orlen\OrlenPaczka\Repository\SelectedPointRepository;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Orlen\OrlenPaczka\Translator\Adapter\Translator;

/**
 * @Route(path="/order", name="orlenpaczka.order")
 */
class OrderController extends FrameworkBundleAdminController
{
    private Translator $translator;

    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
    }

    /**
     * @Route("/generate_label/{id}", name=".generate_label", methods={"POST"})
     */
    public function generateLabelAction(
        $id,
        Request $request,
        GenerateLabelSettings $settings,
        LabelGenerator $generatorLabelFromPackage,
        GenerateLabelOrderFormFactory $generateLabelOrderFormFactory,
        TranslateResult $translateResult
    ): Response
    {
        $form =$generateLabelOrderFormFactory->factory($id);
//        $form =$this->createForm(GenerateLabelOrderType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            try{
                $data = $form->getData();
                $providerData = [
                    'id_packages' => [$id],
                    'form_data' => [
                        'EMail' => $data['EMail'],
                        'FirstName' => $data['FirstName'],
                        'LastName' => $data['LastName'],
                        'CompanyName' => $data['CompanyName'],
                        'PhoneNumber' => $data['PhoneNumber'],
                        'City' => $data['City'],
                        'PostCode' => $data['PostCode'],
                        'BoxSize' => $data['BoxSize'],
                        'DestinationCode' => $data['DestinationPoint'],
                    ],
                ];
                if (isset($data['SenderOrders'])) {
                    $providerData['form_data']['SenderOrders'] = $data['SenderOrders'];
                }
                $settings->setSettings([
                    'provider' => $providerData,
                    'format' => $data['format'],
                    'PrintType' => PrintTypeEnum::ANONYMOUS_ADDRESS,
                    'PrintAdress' => PrintAdressEnum::SENDER_ADDRESS,
                    'allow_regenerate' => true
                ]);


                $generatorLabelFromPackage->generateLabel($settings);
                $translateResult->translateResultOrder($generatorLabelFromPackage->getResult());
                if ($translateResult->hasErrors() == 0 && $translateResult->hasWarnings() == 0 ) {
                    $this->addFlash('success', $this->translator->trans('Generated label', [], 'Modules.Orlenpaczka.Notification'));
                }else{
                    foreach ($translateResult->getSuccess() as $success) {
                        $this->addFlash('success', $success);
                    }
                    foreach ($translateResult->getWarnings() as $warning) {
                        $this->addFlash('warning', $warning);
                    }
                    foreach ($translateResult->getErrors() as $error) {
                        $this->addFlash('error', $error);
                    }
                }

            }
            catch (NoPackageException $e) {
                $this->addFlash('error', $this->translator->trans('Error generate label', [], 'Modules.Orlenpaczka.Notification'));
                $this->addFlash('error', $this->translator->trans('The order already contains a generated shipment', [], 'Modules.Orlenpaczka.Notification'));
            }catch (\Exception $e){
                $this->addFlash('error', $this->translator->trans('Error generate label', [], 'Modules.Orlenpaczka.Notification'));
                $this->addFlash('error', $e->getMessage());
            }
        }
        return $this->redirectToRoute(
            'admin_orders_view', [
            'orderId' => $form->getData()['id_order']
        ]);
    }

    public function generateLabelListAction(
        Request $request,
        GenerateLabelSettings $settings,
        LabelGenerator $generatorLabelFromPackage,
        PackageRepository $packageRepository,
        TranslateResult $translateResult
    )
    {
        $form = $this->createForm(GenerateLabelType::class, [], [
            'url_action' => 'orlenpaczka.order.generate_label_list']);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $data =$form->getData();
            $packagesIds = $packageRepository->getPackagesFromOrderToGenerateLabel($data['packageIds']);
            if (empty($packagesIds)){
                $this->addFlash('error', $this->translator->trans('No packages to generate label', [], 'Modules.Orlenpaczka.Notification'));
                return $this->redirectToRoute(
                    'admin_orders_index' );
            }
            try{
                $data = $form->getData();
                $providerData = [
                    'id_packages' => $packagesIds,

                ];

                $settings->setSettings([
                    'provider' => $providerData,
                    'format' => $data['format'],
                    'PrintType' => PrintTypeEnum::ANONYMOUS_ADDRESS,
                    'PrintAdress' => PrintAdressEnum::SENDER_ADDRESS,
                ]);
                $generatorLabelFromPackage->generateLabel($settings);

                $translateResult->translateResultOrder($generatorLabelFromPackage->getResult());
                if ($translateResult->hasErrors() == 0 && $translateResult->hasWarnings() == 0 ) {
                    $this->addFlash('success', $this->translator->trans('Generated label', [], 'Modules.Orlenpaczka.Notification'));
                }else{
                    foreach ($translateResult->getSuccess() as $success) {
                        $this->addFlash('success', $success);
                    }
                    foreach ($translateResult->getWarnings() as $warning) {
                        $this->addFlash('warning', $warning);
                    }
                    foreach ($translateResult->getErrors() as $error) {
                        $this->addFlash('error', $error);
                    }
                }

            }
            catch (NoPackageException $e) {
                $this->addFlash('error', $this->translator->trans('Error generate label', [], 'Modules.Orlenpaczka.Notification'));
                $this->addFlash('error', $this->translator->trans('No packages to generate label (packages are in delivery or have already been generated label)', [], 'Modules.Orlenpaczka.Notification'));
            }catch (\Exception $e){
                $this->addFlash('error', $this->translator->trans('Error generate label', [], 'Modules.Orlenpaczka.Notification'));
                $this->addFlash('error', $e->getMessage());
            }


        }

        return $this->redirectToRoute(
            'admin_orders_index' );
    }
    /**
     * @Route("/save-pickup", name=".save_pickup", methods={"POST"})
     */
    public function savePickupAction(Request $request, PackageSelectedPointRepository $packageSelectedPointRepository, PackageRepository $packageRepository): Response
    {
        $warning = false;
        $selectedPoint = $packageSelectedPointRepository->findOneBy(['destinationCode' => $request->get('destinationCode')]);
        if (empty($selectedPoint)) {
            $selectedPoint = new PackageSelectedPoint();
        }
        try{
            $selectedPoint->setPoint($request->get('id'));
            $selectedPoint->setName($request->get('name'));
            $selectedPoint->setDestinationCode($request->get('destinationCode'));
            $selectedPoint->setAddress($request->get('address'));
            $selectedPoint->setCity($request->get('city'));
            $selectedPoint->setPostalCode($request->get('postalCode'));
            $selectedPoint->setProvince($request->get('province'));
            $selectedPoint->setLatitude((float)$request->get('latitude'));
            $selectedPoint->setLongitude((float)$request->get('longitude'));
            $packageSelectedPointRepository->add($selectedPoint);
            $packageId = $request->get('package_id');
            $package = $packageRepository->find($packageId);
            if (!empty($package) && $package->getLabelName() === null){
                $package->setPickup($selectedPoint);
                $packageRepository->add($package);
            }elseif (!empty($package) && $package->getPickup() && $package->getPickup()->getDestinationCode() !== $selectedPoint->getDestinationCode()){
                $warning = $this->translator->trans('Destination code  is different from the point on the label', [], 'Modules.Orlenpaczka.Tab');
            }

        }catch (\Exception $e){
            return new JsonResponse(['success' => false, 'error' => $this->translator->trans('Error save Pickup', [], 'Modules.Orlenpaczka.Tab')]);
        }

        return new JsonResponse(['success' => true, 'warning' => $warning, 'message' => $this->translator->trans('Pickup save successfully', [], 'Modules.Orlenpaczka.Tab')]);

    }
    /**
     * @Route("/duplicate/{idOrder}/{id}", name=".duplicate", methods={"GET"})
     */
    public function duplicatePackageAction($id, $idOrder, PackageRepository $packageRepository)
    {
        $result = $packageRepository->duplicateForOrder($id);
        if ($result) {
            $this->addFlash('success', $this->translator->trans('New Package created', [], 'Modules.Orlenpaczka.Tab'));
        }else{
            $this->addFlash('error', $this->translator->trans('Package not created', [], 'Modules.Orlenpaczka.Tab'));
        }
        return $this->redirectToRoute(
            'admin_orders_view', [
            'orderId' => $idOrder
        ]);
    }
}
